/* ===========================================================================
 * Modul: zg2s.c
 *
 * Version 2010-07-21
 *
 * Copyright (C) 2005-2010 LANUV  NRW, Germany
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * ---------------------------------------------------------------------------
 *
 * Determination of slope by given terrain height
 * Read the file readme_eng.txt or readme_ger.txt
 * to get infos about this program.
 * 
 * ---------------------------------------------------------------------------
 *
 * Changelog
 *
 * 2010-07-21
 *  Input formats Atkis and ArcInfo are additionally supported.
 *  Output formats table and ArcInfo are additionally supported.
 * ==========================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ctype.h>
#include "zg2s.h"

int fileRead(void);
int fileWrite(void);
void headerInit(void);
int argRead(int argc, char *argv[]);
int dmnaFileRead(void);
int dmnaFileWrite(void);
int arcInfoFileRead(void);
int arcInfoFileWrite(void);
int atkisFileRead(void);
int tableFileWrite(void);
void ofsNameCreate(void);
int matrixCheck(void);
int cntField(const char *s, const char *t);
void slopeCalc(void);
int getMem(void);
void freeMem(void);
void msg(int rc);
#if defined __unix__ || __linux__
char* _strlwr(char *s);
char* _strupr(char *s);
#endif

static const char 
   info[] = "\n"
      "zg2s, version 2010-07-21\n"
      "Determination of slope by given terrain height\n"
      "Copyright (C) 2005-2010  LANUV NRW, Germany\n"
      "zg2s comes with ABSOLUTELY NO WARRANTY\n"
      "zg2s is free software under the GNU PUBLIC LICENCE\n",
   usage[] =
      "\nUsage         zg2s <file> [<options>]\n"
      "<file>        input file of type ArcInfo, DMNA or Atkis\n"
      "<options>\n"
      "-f<a|d|t>     output format: a=ArcInfo, d=DMNA (default), t=table\n"
      "-o<file name> output file name, default: <input file name>_s\n";

static char src[FSLEN], dst[FSLEN];
static RH **matrix;
static struct header h;

static int
   xDim = 0, yDim = 0,
   indI, indJ,
   xSign = 1, ySign = 1,
   *p1, *p1Sign, *p1Dim,
   *p2, *p2Sign, *p2Dim,
   gkFl = 0, refFl = 0,
   ifsFormat = NON_FORMAT,
   ofsFormat = OFS_DMNA,
   ncFl = 0, prec,
   nrows, ncols;

static double
   xMin, yMin,
   xll, yll, cellsize, nodata_value = -9999.0,
   slopeMax = 0;


int main(int argc, char *argv[])
{
   int rc = 0;

   msg(MSG_INFO);
   if(!(rc = argRead(argc, argv)))
   {
      headerInit();
      if(!(rc = fileRead()))
      {
         slopeCalc();
         rc = fileWrite();
         freeMem();
      }
   }
   msg(rc);
   return rc;
}

int fileRead(void)
{
   int rc = 0;

   rc = atkisFileRead();
   if(ifsFormat == NON_FORMAT && rc == MSG_ATKIS_TRIPLET)
   {
      rc = arcInfoFileRead();
      if(ifsFormat == NON_FORMAT)
         rc = dmnaFileRead();
      if(!rc && ifsFormat == NON_FORMAT)
         rc = MSG_IFS_FORMAT;
   }
   return rc;
}

int fileWrite(void)
{
   int rc = 0;

   ofsNameCreate();
   switch(ofsFormat)
   {
      case OFS_DMNA:
         rc = dmnaFileWrite();
         break;
      case OFS_TABLE:
         rc = tableFileWrite();
         break;
      case OFS_ARCINFO:
         rc = arcInfoFileWrite();
         break;
   }
   return rc;
}

void headerInit(void)
{
   h.lowb[0] = h.lowb[1] = -1;
   h.hghb[0] = h.hghb[1] = -1;
   h.dims = 2;
   h.size = 4;
   h.xmin = 0.0;
   h.ymin = 0.0;
   h.refx = 0.0;
   h.refy = 0.0;
   h.gakrx = 0.0;
   h.gakry = 0.0;
   h.delta = 0.0;
   strncpy(h.locl, "C", VSLEN-1);
   h.locl[VSLEN-1] = '\0';
   h.ggcs[0] = '\0';
   strncpy(h.formatSlope, "%4.0f", VSLEN-1);
   h.formatSlope[VSLEN-1] = '\0';
   strncpy(h.formatSlope2, "%4.0lf", VSLEN-1);
   h.formatSlope2[VSLEN-1] = '\0';
   strncpy(h.artp, "T", VSLEN-1);
   h.artp[VSLEN-1] = '\0';
   strncpy(h.vldf, "P", VSLEN-1);
   h.vldf[VSLEN-1] = '\0';
   strncpy(h.mode, "text", VSLEN-1);
   h.mode[VSLEN-1] = '\0';
   strncpy(h.unit, "cm/m", VSLEN-1);
   h.unit[VSLEN-1] = '\0';
   strncpy(h.name, "Slope", NSLEN-1);
   h.name[NSLEN-1] = '\0';
   h.ident[0] = '\0';
}

int argRead(int argc, char *argv[])
{
   int  c, rc = 0, slen;

   if(argc == 1)
   {
      rc = MSG_USAGE;
   }
   else
   {
      if(2 <= argc && argc <= 4)
      {
         while(--argc > 0 && !rc)
         {
            c = (*++argv)[0];
            if(c == '-' || c == '/')
            {
               if((slen = strlen(*argv)) >= 2)
               {
                  c = tolower((*argv)[1]);
                  switch(c)
                  {
                     case 'f':
                        if(slen == 3)
                        {
                           switch((*argv)[2])
                           {
                              case 'a':
                                 ofsFormat = OFS_ARCINFO;
                                 break;
                              case 'd':
                                 ofsFormat = OFS_DMNA;
                                 break;
                              case 't':
                                 ofsFormat = OFS_TABLE;
                                 break;
                              default:
                                 rc = MSG_OFS_FORMAT;
                           }
                        }
                        else
                           rc = MSG_OFS_FORMAT;
                        break;
                     case 'o':
                        if(slen >= 3)
                        {
                           strncpy(dst, *argv+2, FSLEN-1);
                           dst[FSLEN-1] = '\0';
                           ncFl = 1;
                        }
                        else
                           rc = MSG_NON_OFS;
                        break;
                     default:
                        rc = MSG_LETTER_UNKNOWN;
                  }
               }
               else
                  rc = MSG_LETTER_UNKNOWN;
            }
            else
            {
               strncpy(src, *argv, FSLEN-1);
               src[FSLEN-1] = '\0';
            }
         }
         if(!strlen(src))
            rc = MSG_NON_IFS;
      }
   }
   return rc;
}

int dmnaFileRead(void)
{
   int  i, eoh = 0, rc = 0;
   FILE *fs;
   const char sep[] = " \t\"";
   char *p, s[512] = "",
        artp[VSLEN] = "", vldf[VSLEN] = "", sequ[VSLEN] = "";
   
   if((fs = fopen(src, "r")) == NULL)
      return MSG_IFS;

   while(fgets(s, sizeof(s), fs))
   {
      if(strchr(s, '*') != NULL)
      {
         for(i = 0, p = s; *p; ++p)
            if(!isspace(*p))
               ++i;
         if(i == 1)
         {
            eoh = 1;
            break;
         }
      }
      if((p = strchr(s, '\n')) != NULL)
         *p = '\0';
      p = s;
      while(isspace(*p))
         ++p;
      if(*p == '-')
         continue;
      _strlwr(s);
      if((p = strchr(s, '\'')) != NULL)
         *p = '\0';
      if((p = strstr(s, "sequ")) != NULL)
      {
         if((p = strtok(p+4, sep)) != NULL)
         {
            while(isspace(*p))
               ++p;
            strncpy(sequ, p, VSLEN-1);
            sequ[VSLEN-1] = '\0';
            _strlwr(sequ);
            if(strlen(sequ) >= 5 && h.dims == 2)
            {
               strncpy(h.sequ, sequ, VSLEN-1);
               h.sequ[VSLEN-1] = '\0';
               if(p[0] == 'j' && p[3] == 'i')
               {
                  ySign = p[1] == '-' ? -1 : 1;
                  xSign = p[4] == '-' ? -1 : 1;
                  p1 = &indJ;
                  p1Sign = &ySign;
                  p1Dim = &yDim;
                  p2 = &indI;
                  p2Sign = &xSign;
                  p2Dim = &xDim;
               }
               else
                  if(p[0] == 'i' && p[3] == 'j')
                  {
                     xSign = p[1] == '-' ? -1 : 1;
                     ySign = p[4] == '-' ? -1 : 1;
                     p1 = &indI;
                     p1Sign = &xSign;
                     p1Dim = &xDim;
                     p2 = &indJ;
                     p2Sign = &ySign;
                     p2Dim = &yDim;
                  }
                  else
                     return MSG_SEQU;
            }
            else
               return MSG_SEQU;
         }
      }
      if((p = strstr(s, "artp")) != NULL)
      {
         if((p = strtok(p+4, sep)) != NULL)
         {
            strncpy(artp, p, VSLEN-1);
            artp[VSLEN-1] = '\0';
            _strupr(artp);
         }
      }
      if((p = strstr(s, "vldf")) != NULL)
      {
         if((p = strtok(p+4, sep)) != NULL)
         {
            strncpy(vldf, p, VSLEN-1);
            vldf[VSLEN-1] = '\0';
            _strupr(vldf);
         }
      }
      if((p = strstr(s, "xmin")) != NULL)
      {
         if((p = strtok(p+4, sep)) != NULL)
            h.xmin = atof(p);
      }
      if((p = strstr(s, "ymin")) != NULL)
      {
         if((p = strtok(p+4, sep)) != NULL)
            h.ymin = atof(p);
      }
      if((p = strstr(s, "delta")) != NULL)
      {
         if((p = strtok(p+5, sep)) != NULL)
            h.delta = atof(p);
      }
      if((p = strstr(s, "refx")) != NULL)
      {
         refFl = 1;
         if((p = strtok(p+4, sep)) != NULL)
            h.refx = atof(p);
      }
      if((p = strstr(s, "refy")) != NULL)
      {
         refFl = 1;
         if((p = strtok(p+4, sep)) != NULL)
            h.refy = atof(p);
      }
      if((p = strstr(s, "gakrx")) != NULL)
      {
         gkFl = 1;
         if((p = strtok(p+5, sep)) != NULL)
            h.gakrx = atof(p);
      }
      if((p = strstr(s, "gakry")) != NULL)
      {
         gkFl = 1;
         if((p = strtok(p+5, sep)) != NULL)
            h.gakry = atof(p);
      }
      if((p = strstr(s, "ggcs")) != NULL)
      {
         if((p = strtok(p+4, sep)) != NULL)
         {
            strncpy(h.ggcs, p, VSLEN-1);
            h.ggcs[VSLEN-1] = '\0';
            _strupr(h.ggcs);
         }
      }
      if((p = strstr(s, "form")) != NULL)
      {
         if((p = strtok(p+4, sep)) != NULL)
         {
            char *q;
            strncpy(h.formatHeight, p, VSLEN-1);
            h.formatHeight[VSLEN-1] = '\0';
            strncpy(h.formatHeight2, p, VSLEN-1);
            if((q = strchr(h.formatHeight2, 'f')) != NULL)
               strncpy(q, "lf", 2);
            h.formatHeight2[VSLEN-1] = '\0';
         }
      }
      if((p = strstr(s, "lowb")) != NULL)
      {
         sscanf(p, "%*s%d%d", &h.lowb[0], &h.lowb[1]);
      }
      if((p = strstr(s, "hghb")) != NULL)
      {
         sscanf(s, "%*s%d%d", &h.hghb[0], &h.hghb[1]);
      }
   }
   if(eoh)
   { 
      if(refFl || !gkFl)
      {
         xMin = h.refx + h.xmin + h.lowb[0]*h.delta;
         yMin = h.refy + h.ymin + h.lowb[1]*h.delta;
      }
      else
      {
         xMin = h.gakrx + h.xmin + h.lowb[0]*h.delta;
         yMin = h.gakry + h.ymin + h.lowb[1]*h.delta;
      }
      if(!strlen(h.sequ))
         return MSG_SEQU;
      if(h.delta <= 0.0)
         return MSG_DELTA;
      if(strcmp(h.artp, artp) || strcmp(h.vldf, vldf))
         return MSG_TERRAIN;
      if(!strlen(h.formatHeight))
      {
         strncpy(h.formatHeight, "%7.1f", VSLEN-1);
         h.formatHeight[VSLEN-1] = '\0';
         strncpy(h.formatHeight2, "%7.1lf", VSLEN-1);
         h.formatHeight2[VSLEN-1] = '\0';
      }
      xDim = h.hghb[0] - h.lowb[0] + 1;
      yDim = h.hghb[1] - h.lowb[1] + 1;
      if(xDim > 2 && yDim > 2)
      {
         rc = getMem();
         if(rc != MSG_MEM)
         {
            for(*p1 = *p1Sign > 0 ? 0 : *p1Dim-1; *p1 < *p1Dim && *p1 >= 0 && !rc; *p1 += *p1Sign)
            {
               for(*p2 = *p2Sign > 0 ? 0 : *p2Dim-1; *p2 < *p2Dim && *p2 >= 0 && !rc; *p2 += *p2Sign)
               {
                  if(fscanf(fs, "%s", s) != EOF)
                     matrix[indI][indJ].height = atof(s);
                  else
                     rc = MSG_DMNA_CNT;
               }
            }
            if(!rc)
            {
               ifsFormat = IFS_DMNA;
               rc = matrixCheck();
            }
         }
      }
      else
         rc = MSG_LHB;
   }
   else
      rc = MSG_DMNA_HEADER;
   fclose(fs);
   return rc;
}
 
int dmnaFileWrite(void)
{
   FILE *fd;
   char *p, format[VSLEN] = "";

   if((fd = fopen(dst, "w")) == NULL)
      return MSG_OFS;
   /* Header */
   strncpy(h.sequ, "j-,i+", VSLEN-1);
   h.sequ[VSLEN-1] = '\0';
   fprintf(fd, "locl\t\"%s\"\nxmin\t%.1lf\nymin\t%.1lf\ndelta\t%.1lf\n",
               h.locl, h.xmin, h.ymin, h.delta);
   if(refFl || !gkFl)   
      fprintf(fd, "refx\t%.1lf\nrefy\t%.1lf\n", h.refx, h.refy);
   else
      fprintf(fd, "gakrx\t%.1lf\ngakry\t%.1lf\n", h.gakrx, h.gakry);
   if(strlen(h.ggcs))
      fprintf(fd, "ggcs\t\"%s\"\n", h.ggcs);
   fprintf(fd, "unit\t%s\nname\t%s\nform\t\"%s\"\nartp\t\"%s\"\n"
               "vldf\t\"%s\"\nsequ\t\"%s\"\nmode\t\"%s\"\ndims\t%d\n"
               "size\t%d\nlowb\t%d\t%d\nhghb\t%d\t%d\n*\n",
               h.unit, h.name, h.formatSlope, h.artp,
               h.vldf, h.sequ, h.mode, h.dims, h.size,
               h.lowb[0]+1, h.lowb[1]+1, h.hghb[0]-1, h.hghb[1]-1);
   p = h.sequ;
   if(p[0] == 'j' && p[3] == 'i')
   {
      ySign = p[1] == '-' ? -1 : 1;
      xSign = p[4] == '-' ? -1 : 1;
      p1 = &indJ;
      p1Sign = &ySign;
      p1Dim = &yDim;
      p2 = &indI;
      p2Sign = &xSign;
      p2Dim = &xDim;
   }
   else
      if(p[0] == 'i' && p[3] == 'j')
      {
         xSign = p[1] == '-' ? -1 : 1;
         ySign = p[4] == '-' ? -1 : 1;
         p1 = &indI;
         p1Sign = &xSign;
         p1Dim = &xDim;
         p2 = &indJ;
         p2Sign = &ySign;
         p2Dim = &yDim;
      }
   p = h.formatSlope2;
   format[0] = '\t';
   strncpy(format+1, p, VSLEN-2);
   format[VSLEN-1] = '\0';
   for(*p1 = *p1Sign > 0 ? 1 : *p1Dim-2; *p1 < *p1Dim-1 && *p1 >= 1; *p1 += *p1Sign)
   {
      for(*p2 = *p2Sign > 0 ? 1 : *p2Dim-2; *p2 < *p2Dim-1 && *p2 >= 1; *p2 += *p2Sign)
      {
         fprintf(fd, format, matrix[indI][indJ].slope * 100.0);  /* cm per m */
      }
      fprintf(fd, "\n");
   }
   fprintf(fd, "\n***\n");
   fclose(fd);
   return 0;
}

int arcInfoFileRead(void)
{
   int  i, j, rc = 0, pr = 0;
   FILE *fs;
   char ncs[8] = "", ncsv[16] = "",
        nrs[8] = "", nrsv[16] = "",
        xs[16] = "", xsv[16] = "",
        ys[16] = "", ysv[16] = "",
        cs[16] = "",  csv[16] = "",
        nods[16] = "", nodsv[16] = "",
        *pos, s[80];

   if((fs = fopen(src, "r")) == NULL)
      return MSG_IFS;
   // Header einlesen
   if(fscanf(fs, "%5s%s\n%5s%s\n%9s%s\n%9s%s\n%8s%s\n%12s%s\n",
         ncs, ncsv,
         nrs, nrsv,
         xs, xsv,
         ys, ysv,
         cs, csv,
         nods, nodsv) != 12)
      rc = MSG_ARCINFO_HEADER;
   else
   {
      _strlwr(ncs);
      _strlwr(nrs);
      _strlwr(xs);
      _strlwr(ys);
      _strlwr(cs);
      _strlwr(nods);
      if(strcmp(ncs, "ncols") || strcmp(nrs, "nrows") ||
         (strcmp(xs, "xllcenter") && strcmp(xs, "xllcorner")) ||
         (strcmp(ys, "yllcenter") && strcmp(ys, "yllcorner")) ||
         strcmp(cs, "cellsize") || strcmp(nods, "nodata_value"))
         rc = MSG_ARCINFO_HEADER;
      else
      {
         if((pos = strchr(ncsv, '.')) != NULL)
            *pos = '\0';
         ncols = atoi(ncsv);
         xDim = ncols;
         if((pos = strchr(nrsv, '.')) != NULL)
            *pos = '\0';
         nrows = atoi(nrsv);
         yDim = nrows;
         xll = atof(xsv);
         yll = atof(ysv);
         cellsize = atof(csv);
         nodata_value = atof(nodsv); 
         if(!strcmp(xs, "xllcenter"))
            if(!strcmp(ys, "yllcenter"))
            {
               xll -= cellsize * 0.5;
               yll -= cellsize * 0.5;
            }
            else
               rc = MSG_ARCINFO_CENTER;
         if(!rc)
         {
            if(!(rc = getMem()))
            {
               for(j = nrows-1; j >= 0 && !rc; --j)
               {
                  for(i = 0; i < ncols && !rc; ++i)
                  {
                     if(fscanf(fs, "%s", s) != EOF)
                     {
                        pr = 0;
                        if((pos = strchr(s, '.')) != NULL)
                        {
                           pr = strchr(pos, '\0') - pos - 1;
                           if(prec < pr)
                              prec = pr;
                        }
                        matrix[i][j].height = atof(s);
                     }
                     else
                        rc = MSG_ARCINFO_CNT;
                  }
               }
               ifsFormat = IFS_ARCINFO;
               h.delta = cellsize;
               xMin = h.refx = xll + cellsize * 0.5;
               yMin = h.refy = yll + cellsize * 0.5;
               sprintf(h.formatHeight, "%%7.%df", prec);
               sprintf(h.formatHeight2, "%%7.%dlf", prec);
               h.lowb[0] = 0;
               h.lowb[1] = 0;  
               h.hghb[0] = ncols-1;
               h.hghb[1] = nrows-1;
               if(!rc)
                  rc = matrixCheck();
            }
         }
      }
      fclose(fs);
   }
   return rc;
}

int arcInfoFileWrite(void)
{
   int  i, j, rc = 0;
   char format[VSLEN] = "";
   FILE *fd;

   if((fd = fopen(dst, "w")) == NULL)
      return MSG_OFS;
   strncpy(format, h.formatSlope2, VSLEN-1);
   format[VSLEN-1] = '\0';
   /* Header */
   ncols = h.hghb[0] + 1;
   nrows = h.hghb[1] + 1;
   cellsize = h.delta;
   xll = xMin + cellsize;
   yll = yMin + cellsize;
   fprintf(fd, "ncols        %d\nnrows        %d\n"
      "xllcenter    %.1lf\nyllcenter    %.1lf\n"
      "cellsize     %.1lf\nNODATA_value %.0lf\n",
      ncols-2, nrows-2, xll, yll, cellsize, nodata_value);
   for(j = nrows-2; j >= 1 ; --j)
   {
      for(i = 1; i < ncols-1; ++i)
         fprintf(fd, format, matrix[i][j].slope * 100.0);  /* cm per m */
      fprintf(fd, "\n");
   }
   fclose(fd);
   return rc;
}

int atkisCmp(const void *a, const void *b)
{
   int rc;

   if(((AT*)a)->x < ((AT*)b)->x)
      rc = -1;
   else
   {
      if(((AT*)a)->x > ((AT*)b)->x)
         rc = 1;
      else
      {
         if(((AT*)a)->y < ((AT*)b)->y)
            rc = -1;
         else
         {
            if(((AT*)a)->y > ((AT*)b)->y)
               rc = 1;
            else
               rc = 0;
         }
      }
   }
   return rc;
}

int atkisFileRead(void)
{
   int    i, j, ind, cnt = 0,
          rc = 0, pr = 0;
   double x, y, height, xmin, xmax, ymin; /*, ymax; */
   FILE   *fs;
   char   *pos, s[128];
   AT     *vek;

   if((fs = fopen(src, "r")) == NULL)
      return MSG_IFS;
   while(!rc && fgets(s, sizeof(s), fs))
   {
      if(cntField(s, " \t\n") == 3)
      {
         sscanf(s, "%lf%lf%lf", &x, &y, &height);
         if((pos = strrchr(s, '.')) != NULL)
         {
            pr = strchr(pos, '\0') - pos - 2;
            if(prec < pr)
               prec = pr;
         }
         ++cnt;
      }
      else
         rc = MSG_ATKIS_TRIPLET;
   }
   fclose(fs);
   if(!rc)
   {
      if((vek = (AT *)calloc(cnt, sizeof(AT))) == NULL)
         rc = MSG_MEM;
      else
      {
         fs = fopen(src, "r");
         ind = 0;
         while(fgets(s, sizeof(s), fs))
         {
            sscanf(s, "%lf%lf%lf", &vek[ind].x, &vek[ind].y, &vek[ind].height);
            ++ind;
         }
         fclose(fs);
         qsort(vek, cnt, sizeof(AT), atkisCmp);
         xmin = vek[0].x;
         ymin = vek[0].y;
         xmax = vek[cnt-1].x;
/*       ymax = vek[cnt-1].y; */
         for(ind = 0; ind < cnt-1 && vek[ind].x == vek[ind+1].x; ++ind)
            ;
         yDim = ind + 1;
         if((cellsize = vek[ind+1].x - vek[0].x) != (vek[1].y - vek[0].y))
            rc = MSG_DELTA;
         else
         {
            xDim = (int)((xmax - xmin) / cellsize) + 1;
            if(!(rc = getMem()))
            {
               for(ind = 0; ind < cnt; ++ind)
               {
                  i = (int)((vek[ind].x - xmin) / cellsize);
                  j = (int)((vek[ind].y - ymin) / cellsize);
                  matrix[i][j].height = vek[ind].height;
               }
               rc = matrixCheck();
               ifsFormat = IFS_ATKIS;
               h.delta = cellsize;
               xMin = h.refx = xmin;
               yMin = h.refy = ymin;
               sprintf(h.formatHeight, "%%7.%df", prec);
               sprintf(h.formatHeight2, "%%7.%dlf", prec);
               h.lowb[0] = 0;
               h.lowb[1] = 0;  
               h.hghb[0] = xDim - 1;
               h.hghb[1] = yDim - 1;
            }
         }
         free(vek);
      }
   }
   return rc;
}

int tableFileWrite(void)
{
   int    i, j, len = 0, rc = 0;
   double x, y;
   FILE   *fd;
   char   *ph, *ps, format[2*VSLEN] = "%.0lf\t%.0lf";

   if((fd = fopen(dst, "w")) == NULL)
      return MSG_OFS;
   if(!(ph = strchr(h.formatHeight2, '.')))
      ph = h.formatHeight2;
   if(!(ps = strchr(h.formatSlope2, '.')))
      ps = h.formatSlope2;
   len = strlen(ph) + strlen(ps) + strlen(format) + 6;
   if(len <= sizeof(format))
   {
      strcat(format, "\t%");
      strcat(format, ph);
      strcat(format, "\t%");
      strcat(format, ps);
      strcat(format, "\n");
      fprintf(fd, "X\tY\tHEIGHT\tSLOPE\n");
      for(j = 1; j < yDim - 1; ++j)
      {
         y = yMin + h.delta*j;
         for(i = 1; i < xDim - 1; ++i)
         {
            x = xMin + h.delta*i;
            fprintf(fd, format, x, y,
               matrix[i][j].height,
               matrix[i][j].slope * 100.0);  /* cm per m */
         }
      }
   }
   else
      rc = MSG_STR_FORMAT;
   fclose(fd);
   return rc;
}

void ofsNameCreate(void)
{
   char *p;
   
   if(!ncFl)
   {
      strncpy(dst, src, FSLEN-8);
      dst[FSLEN-8] = '\0';
      if((p = strrchr(dst, '.')) != NULL)
         *p = '\0';
      switch(ofsFormat)
      {
         case OFS_DMNA:
            strcat(dst, "_s.dmna");
            break;
         case OFS_TABLE:
            strcat(dst, "_s.txt");
            break;
         case OFS_ARCINFO:
            strcat(dst, "_s.asc");
            break;
      }
   }
}

int matrixCheck(void)
{
   int i, j, rc = 0;

   for(i = 0; i < xDim; ++i)
      for(j = 0; j < yDim; ++j)
         if(matrix[i][j].height == nodata_value)
            rc = MSG_NDV;
   return rc;
}

int cntField(const char *s, const char *t)
{
   int  cnt = -1;
   const char *p1, *p2;

   if(s && t)
   {
      cnt = 0;
      p1 = s + strspn(s, t);
      p2 = p1 + strcspn(p1, t);
      while(p2 > p1)
      {
         ++cnt;
         p1 = p2 + strspn(p2, t);
         if(p1 > p2)
            p2 = p1 + strcspn(p1, t);
      }
   }
   return cnt;
}

void slopeCalc(void)
{
   int    i, j;
   double distance, slope1, slope2;

   distance = h.delta * 2.0;
   for(j = 1; j < yDim - 1; ++j)
   {
      for(i = 1; i < xDim - 1; ++i)
      {
         slope1 = fabs(matrix[i-1][j].height - matrix[i+1][j].height) / distance;
         slope2 = fabs(matrix[i][j-1].height - matrix[i][j+1].height) / distance;
         matrix[i][j].slope = slope1 > slope2 ? slope1 : slope2;
         matrix[i][j].klass =
            (matrix[i][j].slope <= 0.05) ? KL1 : ((matrix[i][j].slope <= 0.2) ? KL2 : KL3);
         if(matrix[i][j].slope > slopeMax)
            slopeMax = matrix[i][j].slope;
      }
   }
}

int getMem(void)
{
   int i, j, rc = 0;

   if((matrix = (RH **)calloc(xDim, sizeof(RH *))) != NULL)
   {
      for(i = 0; i < xDim && !rc; ++i)
      {
         rc = (matrix[i] = (RH *)calloc(yDim, sizeof(RH))) == NULL ? MSG_MEM : 0;
         if(!rc)
            for(j = 0; j < yDim; ++j)
               matrix[i][j].height = nodata_value;
      }
   }
   else
      rc = MSG_MEM;
   return rc;
}

void freeMem(void)
{
   int i;
   
   for(i = 0; i < xDim; ++i)
      if(matrix[i])
         free(matrix[i]);
   if(matrix)
      free(matrix);
}

#if defined __unix__ || __linux__

char* _strlwr(char *s)
{
   char *p = s;

   while(*p = tolower(*p))
      ++p;

   return s;
}

char* _strupr(char *s)
{
   char *p = s;

   while(*p = toupper(*p))
      ++p;

   return s;
}

#endif

void msg(int rc)
{
   switch(rc)
   {
      case MSG_INFO:
         fprintf(stderr, "%s", info);
         break;
      case MSG_USAGE:
         printf("%s", usage);
         break;
      case MSG_IFS:
         fprintf(stderr, "\ninput file not found\n");
         break;
      case MSG_NON_IFS:
         fprintf(stderr, "\ninput file name not defined\n");
         break;
      case MSG_OFS:
         fprintf(stderr, "\noutput file not created\n");
         break;
      case MSG_NON_OFS:
         fprintf(stderr, "\noutput file name not defined\n");
         break;
      case MSG_DELTA:
         fprintf(stderr, "\ninvalid value for cellsize\n");
         break;
      case MSG_SEQU:
         fprintf(stderr, "\ninvalid sequ\n");
         break;
      case MSG_LHB:
         fprintf(stderr, "\ninvalid values for lowb and/or hghb\n");
         break;
      case MSG_MEM:
         fprintf(stderr, "\nmemory allocation failed\n");
         break;
      case MSG_TERRAIN:
         fprintf(stderr, "\ndmna file doesn't contain terrain values\n");
         break;
      case MSG_IFS_FORMAT:
         fprintf(stderr, "\ninvalid input format\n");
         break;
      case MSG_OFS_FORMAT:
         fprintf(stderr, "\ninvalid output format\n");
         break;
      case MSG_ARCINFO_HEADER:
      case MSG_ARCINFO_CENTER:
      case MSG_DMNA_HEADER:
         fprintf(stderr, "\ninvalid header\n");
         break;
      case MSG_DMNA_CNT:
         fprintf(stderr, "\ndmna file: there are too less valid values\n");
         break;
       case MSG_ARCINFO_CNT:
         fprintf(stderr, "\narcinfo file: there are too less valid values\n");
         break;
     case MSG_NDV:
         fprintf(stderr, "\nthere are too less valid values\n");
         break;
      case MSG_STR_FORMAT:
         fprintf(stderr, "\noutput format string too long\n");
         break;
      case MSG_LETTER_UNKNOWN:
         fprintf(stderr, "\ninvalid option\n");
         break;
   }
}
