// =============================================================== eval-vdi.c
//
// Evaluation of validation vdi (data sets of draft VDI 3783/9)
// ============================================================
//
// Copyright (C) Umweltbundesamt, 14191 Berlin, Germany, 2004
// Copyright (C) Janicke Consulting, 26427 Dunum, Germany, 2004
// email: info@austal2000.de
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of
// the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// last change: 2004-10-06 uj
//
//===========================================================================

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#define WND(i,j,k)  ((WNDREC)Wnd[(i)*(Ny+1)*(Nz+1) + (j)*(Nz+1) + (k)])
#define VDI(i)      ((VDIREC)Vdi[i])

#define D2R  0.017453292

static double InterpolateVv( double *, double *, double, int);
static int getU(double, double, double, double *, double *, double *);
static int readWnd(char *, char *);
static int evaluate(int);
  
typedef struct { double z, vx, vy, vz; } WNDREC;

typedef struct { 
  int i; 
  double x, y, z;
  double v[3];
  double w[3];
} VDIREC;

static char Path[256], WndName[256], VdiName[256];
static WNDREC *Wnd;
static VDIREC *Vdi;
static int Nx, Ny, Nz, Nd;
static double Xmin, Ymin, Dd;
static double X1, X2, Y1, Y2, Z1, Z2, WFac, DFac, RefX, RefY, RefZ, RefR;

//------------------------------------------------------------------------- getU
static int getU(double x, double y, double z, double *vx, double *vy, 
double *vz) {
  int n, i, j, l, k, i1, j1, nzmax=199, ja, ia, ka;
  double uu[200], zz[200];
  double da, db, dc, z00, z01, z10, z11, v, v0, v1;
  ia = 1 + (x-Xmin)/Dd;  
  if (ia < 1 || ia > Nx) return -12;
  da = (x-Xmin)/Dd - (ia-1);
  ja = 1 + (y-Ymin)/Dd;  
  if (ja < 1 || ja > Ny)  return -13;
  db = (y-Ymin)/Dd - (ja-1);
  if (Nz > nzmax) {
    printf("*** maximal %d Vertikalintervalle erlaubt!\n", nzmax);
    exit(1);
  }
  for (k=0; k<=Nz; k++) {
    z00 = WND(ia-1, ja-1, k).z;
    z10 = WND(ia  , ja-1, k).z;
    z01 = WND(ia-1, ja  , k).z;
    z11 = WND(ia  , ja  , k).z;
    if (z00 != z01 || z00 != z10 || z00 != z11) {
      printf("*** Auswertung nur fr ebenes Gelnde implementiert!\n");
      exit(1);
    }
    zz[k] = z00 + da*(z10-z00) + db*(z01-z00) + da*db*(z00+z11-z10-z01);
  }
  for (k=1; k<=Nz; k++)
    if (zz[k]-zz[0] >= z) break;
  if (k > Nz) return -14;
  ka = k;
  dc = (z-zz[k-1])/(zz[k] - zz[k-1]);
  for (k=1; k<=Nz; k++) {
    v0 = WND(ia-1, ja, k).vx;
    v1 = WND(ia  , ja, k).vx;
    uu[k] = v0 + da*(v1-v0);
  }
  *vx = InterpolateVv(uu, zz, z, Nz);
  for (k=1; k<=Nz; k++) {
    v0 = WND(ia, ja-1, k).vy;
    v1 = WND(ia, ja  , k).vy;
    uu[k] = v0 + db*(v1-v0);
  }
  *vy = InterpolateVv(uu, zz, z, Nz);
  v0 = WND(ia, ja, ka-1).vz;
  v1 = WND(ia, ja, ka  ).vz;
  *vz = v0 + dc*(v1-v0);
  return 0;       
}

//---------------------------------------------------------------- InterpolateVv
static double InterpolateVv( double *uu, double *zz, double ha, int nh ) {
  double d, d1, d2, h1, h2, h3, p1, p2, p3, u1, u2;
  int ka;
  if (nh < 1)  return 0;
  for (ka=1; ka<=nh; ka++)  if (zz[ka]-zz[0] >= ha)  break;
  if (ka > nh)  return 0;
  p2 = uu[ka];
  d2 = zz[ka] - zz[ka-1];
  h2 = 0.5*(zz[ka-1]+zz[ka]) - zz[0];
  d = (ha - zz[ka-1] + zz[0])/d2;
  if (ka == 1) {
    h1 = -h2;
    d1 = d2;
    p1 = -p2;
  }
  else {
    h1 = 0.5*(zz[ka-2]+zz[ka-1]) - zz[0];
    d1 = zz[ka-1] - zz[ka-2];
    p1 = uu[ka-1];
  }
  u1 = p1 + (p2-p1)*(ha-h1)/(h2-h1);
  if (ka == nh)  return u1;
  p3 = uu[ka+1];
  h3 = 0.5*(zz[ka]+zz[ka+1]) - zz[0];
  u2 = p2 + (p3-p2)*(ha-h2)/(h3-h2);
  return (1-d)*u1 + d*u2;
}


//---------------------------------------------------------------------- readWnd
static int readWnd(char *path, char *name) {
  char sequ[64], s[256], fn[256], buf[4000]="", *pc;
  int i, j, k, l, i1, j1, k1, size;
  double d, x0, y0;
  FILE *f;
  strcpy(Path, path);
  strcpy(WndName, name);
  sprintf(fn, "%s/%s", path, name);
  f = fopen(fn, "r");                           // Datei ffnen
  if (!f) {
    printf("*** Datei \"%s\" kann nicht gelesen werden!\n", fn);
    exit(1);
  }
  size = -1; 
  Nx = -1;
  Ny = -1;
  Nz = -1;
  Xmin = -1.e-9;
  Ymin = -1.e-9;
  Dd =   -1.e-9;
  while(fgets(buf, 4000, f)) {                  // Header berspringen
    for (i=0; i<strlen(buf); i++) buf[i] = tolower(buf[i]);
    if (!strncmp(buf,"sequ", 4)) {
      sprintf(sequ, "%s", buf+5);
    }
    if (!strncmp(buf,"xmin", 4)) {
      sscanf(buf+4, "%lf", &Xmin);  
    }
    if (!strncmp(buf,"ymin", 4)) {
      sscanf(buf+4, "%lf", &Ymin);    
    }
    if (!strncmp(buf,"delta", 5)) {
      sscanf(buf+5, "%lf", &Dd);    
    }
    if (!strncmp(buf,"size", 4)) {
      sscanf(buf+4, "%d", &size);    
    }
    if (!strncmp(buf,"hghb", 4)) {
      sscanf(buf+4, "%d%d%d", &Nx, &Ny, &Nz);    
    }
    if (!strncmp(buf,"lowb", 4)) {
      sscanf(buf+4, "%d%d%d", &i1, &j1, &k1);    
    }
    if (*buf == '*')  break;
  }
  Nx -= i1;
  Ny -= j1;
  Nz -= k1;
  if (*buf != '*') {
    printf("*** Datenteil in Datei \"%s\" nicht gefunden!\n", fn);
    exit(2);
  }
  if (!strstr(sequ, "\"i,j,k\"") && !strstr(sequ, "\"i+,j+,k+\"")) {
    printf("*** sequ \"i,j,k\" erforderlich in Datei \"%s\"!\n", fn);
    exit(2);
  }
  if (size != 16 || Nx < 1 || Ny < 1 || Nz < 1) {
    printf("*** ungltige Dimensionierung oder size!=16 in Datei \"%s\"!\n", fn);
    exit(2);
  }
  if (Xmin == -1.e-9 || Ymin == -1.e-9 || Dd == -1.e-9) {
    printf("*** ungltiges Gitter xmin/ymin/dd in Datei \"%s\"!\n", fn);
    exit(2);
  }
  Wnd = malloc((Nx+1)*(Ny+1)*(Nz+1)*sizeof(WNDREC));
  if (Wnd == NULL) {
    printf("*** Speicher fr Winddaten kann nicht reserviert weren!\n");
    exit(1);
  }
  i = 0;
  j = 0;
  k = -1;
  while(fgets(buf, 4000, f)) {
    if (*buf == '*') break;
    pc = strtok(buf, " \t\r\n");
    if (!pc)  continue;
    k++;
    if (k > Nz) {
      k = 0;
      j++;
    }
    if (j > Ny) {
      j = 0;
      i++;
    }
    for (l=0; l<4; l++) {
      if (!pc) {
        printf("%d.tes Datenelement in (%d,%d,%d) nicht gefunden!\n", 
        l, i, j, k);
        exit(5);
      }
      if (1 != sscanf(pc, "%lf", &d))  {
        printf("%d.tes Datenelement in (%d,%d,%d) nicht lesbar!\n", 
        l, i, j, k);
        exit(4);
      }
      if (l == 0)      WND(i,j,k).z = d;
      else if (l == 1) WND(i,j,k).vx = d;
      else if (l == 2) WND(i,j,k).vy = d;
      else             WND(i,j,k).vz = d;
      pc = strtok(NULL, " \t\r\n");
    }
  }
  if (i != Nx || j != Ny || k != Nz)  {
    printf("Fehler in der Anzahl der Datenzeilen!\n");
    exit(6);
   }
  fclose(f);
  return 0;
}

//---------------------------------------------------------------------- readWnd
static int readVDI(char *path, char *name) {
  char fn[256], buf[4000]="";
  int i, n, rc;
  double x, y, z, vx, vy, vz;
  FILE *f;
  strcpy(Path, path);
  strcpy(VdiName, name);
  sprintf(fn, "%s/%s", path, name);
  f = fopen(fn, "r");                           // Datei ffnen
  if (!f) {
    printf("*** Datei \"%s\" kann nicht gelesen werden!\n", fn);
    exit(1);
  }
  Nd = 0;
  while(fgets(buf, 4000, f)) {
    if (*buf == '#') continue;
    rc = sscanf(buf, "%d %lf %lf %lf %lf %lf %lf", 
        &i, &x, &y, &z, &vx, &vy, &vz);
    if (rc != 7) continue;
    Nd++;
  }
  Vdi = malloc(Nd*sizeof(VDIREC));
  if (Vdi == NULL) {
    printf("*** Speicher fr VDI-Daten kann nicht reserviert weren!\n");
    exit(1);
  }
  fclose(f);
  f = fopen(fn, "r"); 
  n = 0;
  while(fgets(buf, 4000, f)) {
    if (*buf == '#') continue;
    rc = sscanf(buf, "%d %lf %lf %lf %lf %lf %lf", 
        &i, &x, &y, &z, &vx, &vy, &vz);
    if (rc != 7) continue;
    VDI(n).i = i;
    VDI(n).x = x;
    VDI(n).y = y;
    VDI(n).z = z;
    VDI(n).v[0] = vx;
    VDI(n).v[1] = vy;
    VDI(n).v[2] = vz;
    i = n;
    n++;
  }
  fclose(f);
  for (i=strlen(name)-1; i>=0; i--)
    if (name[i] == '.') break;
  name[i] = '\0';
  sprintf(fn, "%s/%s.TXT", path, name);
  f = fopen(fn, "r");    
  if (!f) {
    printf("*** Datei \"%s\" kann nicht gelesen werden!\n", fn);
    exit(1);
  }
  X1 = 0;
  X2 = 0;
  Y1 = 0;
  Y2 = 0;
  Z1 = 0;
  Z2 = 0;
  RefX = 0;
  RefY = 0;
  RefZ = 10;
  WFac = 0;
  DFac = 0;
  while(fgets(buf, 4000, f)) {
    if (*buf == '#') continue;
    if (!strncmp(buf,"xmin", 4)) {
      sscanf(buf+4, "%lf", &X1);
    }
    if (!strncmp(buf,"xmax", 4)) {
      sscanf(buf+4, "%lf", &X2);
    }
    if (!strncmp(buf,"ymin", 4)) {
      sscanf(buf+4, "%lf", &Y1);
    }
    if (!strncmp(buf,"ymax", 4)) {
      sscanf(buf+4, "%lf", &Y2);
    }
    if (!strncmp(buf,"zmin", 4)) {
      sscanf(buf+4, "%lf", &Z1);
    }
    if (!strncmp(buf,"zmax", 4)) {
      sscanf(buf+4, "%lf", &Z2);
    }
    if (!strncmp(buf,"refx", 4)) {
      sscanf(buf+4, "%lf", &RefX);
    }
    if (!strncmp(buf,"refy", 4)) {
      sscanf(buf+4, "%lf", &RefY);
    }
    if (!strncmp(buf,"refz", 4)) {
      sscanf(buf+4, "%lf", &RefZ);
    }
    if (!strncmp(buf,"refr", 4)) {
      sscanf(buf+4, "%lf", &RefR);
      RefR *= D2R;
    }
    if (!strncmp(buf,"wfac", 4)) {
      sscanf(buf+4, "%lf", &WFac);
    }
    if (!strncmp(buf,"dfac", 4)) {
      sscanf(buf+4, "%lf", &DFac);
    }
  }
  fclose(f);
}

//-------------------------------------------------------------------- evaluate
static int evaluate(int list) {
  int n, i, k, agree, rc, nn[3], ok[3];
  char pok[3];
  double x, y, z, wx, wy, wz, si, co, p, o, norm;
  rc = getU(RefX, RefY, RefZ, &wx, &wy, &wz);
  if (rc < 0) return -1;
  norm = -wx*sin(RefR) - wy*cos(RefR);
  if (norm <= 0) return -2;
  for (i=0; i<Nd; i++) {
    rc = getU(VDI(i).x, VDI(i).y, VDI(i).z, &wx, &wy, &wz);
    if (rc < 0) {
      wx = -999;
      wy = -999;
      wz = -999;
    }
    VDI(i).w[0] = wx/norm;
    VDI(i).w[1] = wy/norm;
    VDI(i).w[2] = wz/norm;
  }
  for (k=0; k<3; k++) {
    nn[k] = 0;
    ok[k] = 0;
  }
  co = cos(RefR - 270*D2R);
  si = sin(RefR - 270*D2R);
  if (list)
    printf(" nr index   x(m)   y(m)   z(m)  vdi/vx  vdi/vy  vdi/vz    wnd/vx  wnd/vy  wnd/vz    x y z\n");
  n = 1;
  for (i=0; i<Nd; i++) {
    x = co*VDI(i).x - si*VDI(i).y;
    y = si*VDI(i).x + co*VDI(i).y;
    z = VDI(i).z;
    if (x<X1 || x>X2 || y<Y1 || y>Y2 || z<Z1 || z>Z2)
      continue;
    for (k=0; k<3; k++) {
      p = VDI(i).w[k];
      o = VDI(i).v[k];
      pok[k] = '-';
      if (o <= -99. || p <= -99.) 
        continue;
      if (o != 0 && fabs((p - o)/o) <= DFac)
        agree = 1;
      else if (fabs(p - o) <= WFac)
        agree = 1;
      else
        agree = 0;
      nn[k]++;
      if (agree) ok[k]++;
      pok[k] = (agree) ? '1' : '0';
    }
    if (list) 
      printf("%3d %5d %6.1lf %6.1lf %6.1lf %7.3lf %7.3lf %7.3lf   %7.3lf %7.3lf %7.3lf    %c %c %c\n", 
      n, VDI(i).i, VDI(i).x, VDI(i).y, VDI(i).z, 
      VDI(i).v[0], VDI(i).v[1], VDI(i).v[2],
      VDI(i).w[0], VDI(i).w[1], VDI(i).w[2],
      pok[0], pok[1], pok[2]);
    n++;
  }  
  printf("Ordner            : %s\n", Path);
  printf("Windfelddatei     : %s\n", WndName);
  printf("VDI-Datensatz     : %s\n", VdiName);
  printf("Parameter W       : %4.2f\n", WFac);
  printf("Parameter D       : %4.2f\n", DFac);
  printf("Anstroemung (Grad): %4.0f\n", RefR/D2R);
  printf("Normierung (m/s)  : %4.2f\n", norm);
  printf("x (m) vor Drehung : [%6.1f,%6.1f]\n", X1, X2);
  printf("y (m) vor Drehung : [%6.1f,%6.1f]\n", Y1, Y2);
  printf("z (m)             : [%6.1f,%6.1f]\n", Z1, Z2);
  for (k=0; k<3; k++)
    printf("Trefferquote %c    : %3.0f Prozent (%3d Werte)\n", 
    'u'+k, (nn[k]) ? (float)ok[k]/nn[k]*100. : 0, nn[k]);
  printf("\n");
}


//==================================================================== main

int main( int argc, char *argv[] ) {
  double x, y, z, vx, vy, vz;
  int rc, list;
  if (argc < 4) {
    printf("Aufruf: eval-vdi <Ordner> <Windfelddatei> <VDI-Datensatz> [-l]\n");
    exit(0);
  }
  printf("Test %s:\n", argv[1]);
  Wnd = NULL;
  Vdi = NULL;
  rc = readWnd(argv[1], argv[2]);  
  if (rc < 0) {
    printf("*** Fehler beim Einlesen der Windfelddatei (%d)!\n", rc);
    exit(-1);
  }
  rc = readVDI(argv[1], argv[3]);
  if (rc < 0) {
    printf("*** Fehler beim Einlesen des VDI-Datensatzes (%d)!\n", rc);
    exit(-1);
  }
  list = (argc > 4 && !strncmp(argv[4],"-l", 2)) ? 1 : 0;
  rc = evaluate(list);
  if (rc < 0) {
    printf("*** Fehler bei der Auswertung (%d)!\n", rc);
    exit(-1);
  }
  exit(0);

}

