/*====================================================================== IBJtm.h
 *
 * Time and date routines for IBJ programs
 * =======================================
 *
 * Copyright (C) Janicke Consulting, 88662 berlingen, Germany, 2011-2019
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * last change 2013-10-14 uj
 *
 *============================================================================*/

#include <float.h>
#include <limits.h>

#ifndef IBJTM_INCLUDE
#define	IBJTM_INCLUDE

#define TM_MAX_TIME  DBL_MAX
#define TM_MIN_TIME  -DBL_MAX
#define TM_MAX_DATE  LLONG_MAX
#define TM_MIN_DATE  LLONG_MIN


typedef long long int TMDATE;
typedef double TMTIME;

typedef struct brokendate {
  int year;
  int month;
  int day;
  int hour;
  int minute;
  int second;
  int millisec;
  int tzmin;        // =INT_MIN if unknown
  int leap;
} TMBROKEN;

extern char TmDateSeparator;

int TmParseDate(      // Parse a date of form yyyy-MM-dd[Thh:mm:ss[.SSS][ZZZZZ]]
    TMBROKEN *pbd,    // the parsed date as broken date
    char *s)          // the source string to be parsed
;                     // RETURNS 0 or <0 on error

int TmIsLeapYear(     // Checks for leap year
    int year)         // the year to be checked
;                     // RETURNS 1 if <year> is a leap year

int TmZone(void)      // Get the local time zone
;                     // RETURNS difference to GMT in minutes

TMBROKEN TmDate(void)   // Get the current date
;                       // RETURNS the broken date with local time zone

TMDATE TmGetDate(     // Convert a broken date into date
    TMBROKEN bd)      // the broken date
;                     // RETURNS the date using GMT if time zone is unknown

TMBROKEN TmBreakDate(   // Break a date into components
    TMDATE date,        // the date
    int tzmin)          // the time zone to be used
;                       // RETURNS the broken date

int TmFormatDate(     // Format a date
    char *buf,        // result (buffer size >= 29)
    char *format,     // format as "%<l>[.<p>]D" with 10..l..28, 0..p..3
                      // or NULL for default "%19D"
    TMBROKEN bd)      // the broken date value
;                     // RETURNS <0 on error

int TmHasZone(      // Check the time zone of a broken date
    TMBROKEN *pbd)  // pointer to broken date
;                   // RETURNS != 0 if time zone is valid

int TmSetZone(      // Set (if unknown) or change the time zone of a broken date
    TMBROKEN *pbd,  // pointer to broken date
    int tzmin)      // the new time zone (-720 <= tzmin <= 840)
;                   // RETURNS <0 on error

int TmGetTzShift(   // Get the time shift of a time zone
    char *tmzn)     // time zone as [GMT][+|-]hh[:]mm
;                   // RETURNS the time shift in minutes

int TmParseTime(    // Parse a time string of form [ddd.]hh:mm:ss[.SSS]
    TMTIME *ptm,    // pointer to time value
    char *s)        // the string to be parsed
;                   // RETURNS <0 on error

int TmFormatTime(   // Format a time value as [ddd.]mm:hh:ss[.SSS]
    char *buffer,   // result buffer (size >=24)
    char *format,   // format as "%<l>[.<p>]T" with 0..p..9
    TMTIME tm)      // the time value to be formatted
;                   // RETURNS <0 on error

int TmFormatZone(     // Format the time zone value as "GMT+xx:yy"
    char *buffer,     // the result buffer (size >= 10)
    int tzmin)        // the time zone value (minutes)
;                     // RETURNS 0

TMTIME TmDiffTime(    // Get a time difference
    TMDATE newdate,   // the new date
    TMDATE olddate)   // the old date
;                     // RETURNS the difference new-old

TMDATE TmShiftDate(   // Shift a date by a time span
    TMDATE date,      // the unshifted date
    TMTIME tm)        // the time span
;                     // RETURNS the shifted date

TMTIME TmClock(void)    // Get the current time
;                       // RETURNS the time span from the start of the program

TMDATE TmConvertDate(   // Convert date
    double d)           // date in days as defined in MSG
;                       // RETURNS date in milli-seconds as defined in TM

int TmRelation(               // 1, if data are valid within [*pt1,...]
TMTIME vt1, TMTIME vt2,       // start and end of validity time interval
TMTIME rt1, TMTIME rt2 )      // start and end of requested time interval
;

#endif	/* IBJTM_H */

