// ================================================================= IBJhdr.java
//
// Utility for managing DMN headers
// ================================
//
// Copyright 2005-2011 Janicke Consulting, 88662 Überlingen
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of
// the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// History:
//
// 2007-01-04 format and parse date as well as long
// 2007-02-19 keys not case sensitive
// 2008-01-14  uj  long extensions; getDayOfWeek()
// 2008-12-02  lj  handling of timezone
// 2008-12-05  lj  merged with uj
// 2009-04-21  lj  Properties replaced by LinkedHashMap
// 2010-01-20  lj  parseTime(), parseDate() and parseDateLong() revised
// 2010-01-22  lj  class ZonedDate
// 2010-02-02  lj  contains() added
// 2010-02-16  uj  new function formatValidDate()
// 2010-06-08  uj  catch zero time zone in parseDateLong()
//
// =============================================================================

package de.janicke.ibjutil;

import java.io.PrintWriter;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import java.util.Vector;

/**
 * 
 *  <p>Copyright 2005-2011 Janicke Consulting, 88662 Überlingen</p>
 * 
 * A header for {@link IBJarr IBJarr} containing key/value pairs, a locale
 * and a time zone. Both the keys
 * and the values are strings. Binary data are formatted according to format 
 * definitions supplied to the appropriate methods.  
 * See {@link IBJarr.Descriptor IBJarr.Descriptor} for data of type 
 * <code><i>DATE</i></code> or <code><i>TIME</i></code>.
 *
 * The parameters locale and time zone should be handeled by the specific
 * getter and setter only, not by getString() or setString().
 *
 * @author Lutz Janicke, Janicke Consulting, Dunum
 * @version 2010-06-08
 */
public class IBJhdr {
  
  /** Print check output to {@code System.err} */
  public static boolean CHECK = false;
  public static final long LHOUR = 3600000;
  public static final long LDAY = 86400000;
  public static final String date_format = "yyyy-MM-dd.HH:mm:ss";
  public static final String default_date = "2006-01-01.00:00:00";
  public static final String default_tmzn = "GMT+01:00";
  public static final String default_locl = "C";
  private static final double time_0 = 88609161600000.0;  // 1970-01-01.00:00:00 in ms
  LinkedHashMap<String, String> p;
  TimeZone timezone;
  Locale locale;

  /**
   * Create a void header.
   */
  public IBJhdr() {
    p = new LinkedHashMap<String, String>();
    setLocale(default_locl);
    setTimeZone(default_tmzn);
  }
  
  /**
   * Create a header and copy the parameter definitions, the locale and the
   * time zone from the supplied header.
   * @param hdr header providing definitions.
   */
  public IBJhdr(IBJhdr hdr) {
    this();
    if (hdr == null)  return;
    if (hdr.p != null)  p.putAll(hdr.p);
    setLocale(hdr.getLocale());
    setTimeZone(hdr.getTimeZone().getID());
  }

  public Map<String, String> getMap() {
    return p;
  }
  
  /**
   * Parse an array of strings for doubles.
   * @param ss array of strings to be parsed.
   * @return array of double values or {@code null} in case of parsing error.
   */
  public static double[] parseDoubles(String[] ss) {
    double[] dd = null;
    try {
      int n = ss.length;
      dd = new double[n];
      for (int i = 0; i < n; i++)
        dd[i] = Double.valueOf(ss[i].replace(',', '.').trim());
    } catch (Exception e) {
      return null;
    }
    return dd;
  }
  
  /**
   * Parse a string for a double value.
   * @param s string to be parsed.
   * @return the double value or {@code Double.NaN} in case of parsing error.
   */
  public static double parseDouble(String s) {
    double d = Double.NaN;
    try {
      d = Double.valueOf(s.replace(',', '.').trim());
    } catch (Exception e) {}
    return d;
  }
  
  /**
   * Parse an array of strings for float values.
   * @param ss array of strings to be parsed.
   * @return array of float values or {@code null} in case of parsing error.
   */
  public static float[] parseFloats(String[] ss) {
    float[] ff = null;
    try {
      int n = ss.length;
      ff = new float[n];
      for (int i = 0; i < n; i++)
        ff[i] = Float.valueOf(ss[i].replace(',', '.').trim());
    } catch (Exception e) {
      return null;
    }
    return ff;
  }
  
  /**
   * Parse a string for a float value.
   * @param s string to be parsed.
   * @return the float value or {@code Float.NaN} in case of parsing error.
   */
  public static float parseFloat(String s) {
    float f = Float.NaN;
    try {
      f = Float.valueOf(s.replace(',', '.').trim());
    } catch (Exception e) {}
    return f;
  }
  
  /**
   * Parse an array of strings for integer values.
   * @param ss array of strings to be parsed.
   * @return array of integer values or {@code null} in case of parsing error.
   */
  public static int[] parseIntegers(String[] ss) {
    int[] ii;
    try {
      int n = ss.length;
      ii = new int[n];
      for (int i = 0; i < n; i++)
        ii[i] = Integer.valueOf(ss[i].trim());
    } catch (Exception e) {
      return null;
    }
    return ii;
  }
  
  /**
   * Parse a string for an integer value.
   * @param s string to be parsed.
   * @return the integer value or {@code 0} in case of parsing error.
   */
  public static int parseInteger(String s) {
    int i = 0;
    try {
      i = Integer.valueOf(s.trim());
    }
    catch (Exception e) {}
    return i;
  }

  /**
   * Parse an array of strings for long values.
   * @param ss array of strings to be parsed.
   * @return array of long values or {@code null} in case of parsing error.
   */
  public static long[] parseLongs(String[] ss) {
    long[] ll = null;
    try {
      int n = ss.length;
      ll = new long[n];
      for (int i = 0; i < n; i++)
        ll[i] = Long.valueOf(ss[i].trim());
    } catch (Exception e) {
      return null;
    }
    return ll;
  }
  
  /**
   * Parse a string for a long value.
   * @param s string to be parsed.
   * @return the long value or {@code 0} in case of parsing error.
   */
  public static long parseLong(String s) {
    long l = 0;
    try {
      l = Long.valueOf(s.trim());
    } catch (Exception e) {}
    return l;
  }

  /**
   * Parse an array of strings for time values formatted as DDD.HH:mm:ss.
   * @param ss array of strings to be parsed.
   * @return array of int values (times) or {@code null} in case of parsing error.
   */
  public static int[] parseTimes(String[] ss) {
    int[] ii;
    try {
      int n = ss.length;
      ii = new int[n];
      for (int i = 0; i < n; i++)
        ii[i] = parseTime(ss[i]);
    } catch (Exception e) {
      return null;
    }
    return ii;
  }
  
  /**
   * Parse a string for a time value formatted as DDD.HH:mm:ss.
   * @param s string to be parsed.
   * @return the int value or {@code -1} in case of parsing error.
   */
  public static int parseTime(String s) {
    int time=0, days=0, hours=0, minutes=0, seconds=0;
    if (s == null)                                                //-2010-01-20
      return -1;
    s = s.toLowerCase().trim();
    if (s.startsWith("-inf"))
      return Integer.MIN_VALUE;
    if (s.startsWith("+inf"))
      return Integer.MAX_VALUE;
    try {
      String[] ss = s.split("[.]");
      if (ss.length == 2) {
        days = Integer.valueOf(ss[0]);
        ss[0] = ss[1];
      }
      String[] tt = ss[0].split(":");
      if (tt.length == 3) {
        hours = Integer.valueOf(tt[0]);
        tt[0] = tt[1];
        tt[1] = tt[2];
      }
      if (tt.length >= 2) {
        minutes = Integer.valueOf(tt[0]);
        tt[0] = tt[1];
      }
      seconds = Integer.valueOf(tt[0]);
      if (days < 0 || hours < 0 || minutes < 0 || seconds < 0)    //-2010-01-20
        time = -1;
      else
        time = seconds + 60*(minutes + 60*(hours + 24*days));
    }
    catch (Exception e) {
      if (CHECK) e.printStackTrace();
      time = -1;                                                  //-2010-01-20
    }
    return time;
  }
  
  /**
   * Parse an array of strings for date values formatted as yyyy-MM-dd.HH:mm:ss.
   * @param ss array of strings to be parsed.
   * @param zone the name of the time zone to be used.
   * @return array of double values (dates) or {@code null} in case of parsing error.
   */
  public static double[] parseDates(String[] ss, String zone) {
    if (zone == null)                                             //-2010-01-22
      zone = default_tmzn;
    TimeZone tz = TimeZone.getTimeZone(zone);
    return parseDates(ss, tz);
  }
        
  /**
   * Parse an array of strings for date values formatted as yyyy-MM-dd.HH:mm:ss.
   * @param ss array of strings to be parsed.
   * @param tz the time zone to be used.
   * @return array of double values (dates) or {@code null} in case of parsing error.
   */
  public static double[] parseDates(String[] ss, TimeZone tz) {
    double[] dd;
    try {
      int n = ss.length;
      dd = new double[n];
      for (int i = 0; i < n; i++)
        dd[i] = parseDate(ss[i], tz);
    } catch (Exception e) {
      return null;
    }
    return dd;
  }
  
  /**
   * Parse a string for a date value formatted as yyyy-MM-dd.HH:mm:ss.
   * @param s the strings to be parsed.
   * @param zone the name of the time zone to be used.
   * @return the double value (date) or {@code Double.NaN} in case of parsing error.
   */
  public static double parseDate(String s, String zone) {
    if (zone == null)                                             //-2010-01-22
      zone = default_tmzn;
    TimeZone tz = TimeZone.getTimeZone(zone);
    return parseDate(s, tz);
  }
  
  /**
   * Parse a string for a date value formatted as yyyy-MM-dd.HH:mm:ss.
   * @param s the strings to be parsed.
   * @param tz the time zone to be used.
   * @return the double value (date) or {@code Double.NaN} in case of parsing error.
   */
  public static double parseDate(String s, TimeZone tz) {
    s = s.toLowerCase();
    if (s.startsWith("-inf"))
      return Double.NEGATIVE_INFINITY;
    if (s.startsWith("+inf"))
      return Double.POSITIVE_INFINITY;
    long ms = parseDateLong(s, tz);
    if (ms < 0)
      return Double.NaN;
    double d = (ms + time_0)/(3600*24*1000);
    return d;
  }
  
  /**
   * Parse a string for a date value formatted as yyyy-MM-dd.HH:mm:ss.
   * @param s the strings to be parsed.
   * @param tz the time zone to be used.
   * @return the long value (date) or -1 in case of parsing error.
   */
  public static long parseDateLong(String s, TimeZone tz) {
    s = s.toLowerCase();
    if (s.startsWith("-inf"))
      return Long.MIN_VALUE;
    else if (s.startsWith("+inf"))
      return Long.MAX_VALUE;
    long ms = -1;
    try {
      SimpleDateFormat sdf = new SimpleDateFormat(date_format);   //-2010-01-22
      Date dt = null;
      int l = s.indexOf("GMT");
      if (l > 0) {
        tz = TimeZone.getTimeZone(s.substring(l));
        s = s.substring(0, l).trim();
      }
      else {
        l = s.length();
        if ((s.charAt(l-5) == '-') || (s.charAt(l-5) == '+')) {
          tz = TimeZone.getTimeZone("GMT" + s.substring(l-5));
          s = s.substring(0, l-5).trim();
        }
      }
      if (!s.contains("."))
        s += ".00:00:00";
      sdf.setTimeZone((tz != null) ? tz : TimeZone.getDefault());  //-2010-06-08
      dt = sdf.parse(s);
      ms = dt.getTime();
    }
    catch (Exception e) {
      //e.printStackTrace();
      ms = -1;
    }
    return ms;
  }


  /**
   * Unquote a string.
   * 
   * @param s
   *          the original string
   * @return the unquoted string
   */
  public static String unquote(String s) {
    if (s == null)
      return null;
    int l = s.length();
    if (l < 1 || s.charAt(0) != '\"')
      return s;
    if (s.charAt(l - 1) == '\"')
      l--;
    String t = (l > 1) ? s.substring(1, l) : "";
    return t;
  }
  
  /**
   * Unquote an array of strings.
   * @param ss array of quoted strings
   * @return array of unquoted strings
   */
  public static String[] unquote(String[] ss) {
    if (ss == null)  return null;
    int n = ss.length;
    String[] tt = new String[n];
    for (int i=0; i<n; i++)
      tt[i] = unquote(ss[i]);
    return tt;
  }
  
  /**
   * Format a time value as DDD.HH:mm:ss.
   * @param f time (milliseconds).
   * @return the formatted time.
   */
  public static String formatTime(long f) {
    if (f == Long.MIN_VALUE) return "          -inf";
    if (f == Long.MAX_VALUE) return "          +inf";
    return IBJhdr.formatTime((int)(f/1000));
  }
  
  /**
   * Format a time value as DDD.HH:mm:ss.
   * @param f time (seconds).
   * @return the formatted time.
   */
  public static String formatTime(int f) {
    int sec, min, hour, day;
    char sign;
    if (f < 0) {
      sign = '-';
      f = -f;
    }
    else sign = ' ';
    sec = f % 60;
    f /= 60;
    min = f % 60;
    f /= 60;
    hour = f % 24;
    f /= 24;
    day = f;
    String s = String.format("    %c%d.%02d:%02d:%02d", sign, day, hour, min, sec);
    s = s.substring(s.length()-14);    
    return s;
  }
  
  /**
   * Format a date value as yyyy-MM-dd.HH:mm:ss.
   * @param d the date as double.
   * @param tz the time zone to be used.
   * @return the formatted time.
   */
  public static String formatDate(double d, TimeZone tz) {
    if (d < -1.e-5)
      return "-inf";
    if (d < 0)  d = 0;                
    if (d >= 9999999)
      return "+inf";
    long ms = Math.round(d*3600*24*1000 - time_0);
    Date dt = new Date(ms);
    SimpleDateFormat sdf = new SimpleDateFormat(date_format);
    sdf.setTimeZone(tz);
    String dstr = sdf.format(dt);
    return dstr;
  }

  /**
   * Format a date value as yyyy-MM-dd.HH:mm:ss.
   * @param l the date as long.
   * @param tz the time zone to be used.
   * @return the formatted time.
   */
  public static String formatDate(long l, TimeZone tz) {
    if (l == Long.MIN_VALUE)
      return "-inf";
    if (l == Long.MAX_VALUE)
      return "+inf";
    if (l < 0)  l = Calendar.getInstance().getTimeInMillis();
    if (tz == null)                                               //-2010-01-22
      tz = TimeZone.getTimeZone(default_tmzn);
    SimpleDateFormat sdf = new SimpleDateFormat(date_format);
    sdf.setTimeZone(tz);
    String dstr = sdf.format(new Date(l));
    return dstr;
  }
  
  /**
   * Format a date value as yyyy-MM-dd.HH:mm:ss.
   * @param l the date as long.
   * @param tz the time zone to be used.
   * @return the formatted time.
   */
  public static String formatValidDate(long l, TimeZone tz) {
    if (l <= 0 && l != Long.MIN_VALUE)
      return "                   ";
    if (l == Long.MIN_VALUE)  
      return "-inf";   
    if (l == Long.MAX_VALUE)  
      return "+inf";  
    if (tz == null)
      tz = TimeZone.getTimeZone(default_tmzn);
    SimpleDateFormat sdf = new SimpleDateFormat(date_format);
    sdf.setTimeZone(tz);
    String dstr = sdf.format(new Date(l));
    return dstr;
  }
  
  /**
   * Returns the day of the week (1: Monday to 7: Sunday; or -1)
   * @param t  the date
   * @param tz the time zone
   * @return day-of-week index
   */
  public static int getDayOfWeek(long t, TimeZone tz) {
    Calendar cl = Calendar.getInstance(tz);
    cl.setTimeInMillis(t);
    int d = cl.get(Calendar.DAY_OF_WEEK);
    if (d == Calendar.MONDAY)
      d = 1;
    else if (d == Calendar.TUESDAY)
      d = 2;
    else if (d == Calendar.WEDNESDAY)
      d = 3;
    else if (d == Calendar.THURSDAY)
      d = 4;
    else if (d == Calendar.FRIDAY)
      d = 5;
    else if (d == Calendar.SATURDAY)
      d = 6;
    else if (d == Calendar.SUNDAY)
      d = 7;
    else
      d = -1;
    return d;   
  }
  
  /**
   * Convert date from double to long
   * @param d the date as double.
   * @return the date as long.
   */
  public static long longDate(double d) {  
    return Math.round(d*LDAY - time_0);
  }

  
  /**
   * Convert a <code><i>DATE</i></code> into a {@code java.util.Date}.
   * @param d the date as double.
   * @return the Date.
   */
  public static Date getJDate(double d) {
    long ms = Math.round(d*LDAY - time_0);
    return new Date(ms);   
  }
  
  /**
   * Convert a {@code java.util.Date} into a <code><i>DATE</i></code>.
   * @param dt the Date.
   * @return the date value as double.
   */
  public static double getDate(Date dt) {
    return (dt.getTime() + time_0)/LDAY;    
  }
  
  /**
   * Checks a date value.
   * @param d the date value.
   * @return is plus infinity.
   */
  public static boolean isPlusInfTime(double d) {
    return (d >= 9999999);
  }
  
  /**
   * Checks a date value.
   * @param d the date value.
   * @return is minus infinity.
   */
  public static boolean isNegInfTime(double d) {
    return (d < -1.0e-5);
  }

  /**
   * Tokenize a line with quoted tokens.
   * 
   * @param line
   *          the line to be tokenized.
   * @return the tokens found.
   */
  static public String[] tokenize(String line) {
    if (line == null)
      return null;
    Vector<String> vv = new Vector<String>();
    String[] ss = null;
    int l = line.length();
    for (int i=0; i<l; i++) {
      for (; i < l; i++)
        if (!Character.isWhitespace(line.charAt(i)))
          break;
      if (i >= l)
        break;
      int i1 = i;
      if (line.charAt(i) == '\"') {
        i++;
        for (; i < l; i++) {
          if (line.charAt(i) == '\"') {
            if (line.charAt(i-1) == '\\') continue;
            i++;
            break;
          }
        }
      } 
      else {
        for (; i < l; i++)
          if (Character.isWhitespace(line.charAt(i)))
            break;
      }
      vv.add(line.substring(i1, i));
    }
    ss = vv.toArray(new String[] {});
    return ss;
  }
  
  //-----------------------------------------------------------------------

  public boolean contains(String key) {
    return p.containsKey(key);
  }
  
  /**
   * Set the time zone for formatted dates.
   * @param zone the name of the time zone.
   */
  public void setTimeZone(String zone) {
    if (zone == null)
      zone = default_tmzn;
    timezone = TimeZone.getTimeZone(zone);
    putString("tmzn", timezone.getID(), true);                    //-2008-12-02
  }
  
  /**
   * Get the time zone currently used for formatted dates.
   * @return the time zone.
   */
  public TimeZone getTimeZone() {
    return timezone;
  }
  
  /**
   * Set the locale to be used for formatting floating point numbers. 
   * @param locale the new locale.
   */
  public void setLocale(Locale locale) {
    setLocale((locale == Locale.GERMAN) ? "german" : default_locl);
  }
  
  /**
   * Set the locale to be used for formatting floating point numbers.
   * @param s "C" or "german".
   */
  public void setLocale(String s) {
    if (s == null)
      s = default_locl;
    if (s.equals("german")) {
      locale = Locale.GERMAN;
      putString("locl", "german", true);
    }
    else {
      locale = Locale.ENGLISH;
      putString("locl", default_locl, true);
    }
  }
  
  /**
   * Get the locale currently used for formatting floating point numbers.
   * @return the locale.
   */
  public Locale getLocale() {
    return locale;
  }
  
  /**
   * Get the double values stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the values or <code>null</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public double[] getDoubles(String key) {
    if (key == null)
      return null;
    double[] dd = parseDoubles(getStrings(key));
    return dd;
  }
  
  /**
   * Get the double value stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the value or <code>NaN</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public double getDouble(String key) {
    double d = Double.NaN;
    if (key == null)
      return d;
    String s = getString(key);
    d = parseDouble(s);
    return d;
  }
  
  /**
   * Get the float values stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the values or <code>null</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public float[] getFloats(String key) {
    if (key == null)
      return null;
    String[] ss = getStrings(key);
    float[] ff = parseFloats(ss);
    return ff;
  }
  
  /**
   * Get the float value stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the value or <code>Float.NaN</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public float getFloat(String key) {
    float f = Float.NaN;
    if (key == null)
      return f;
    String s = getString(key);
    f = parseFloat(s);
    return f;
  }
  
  /**
   * Get the integer values stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the values or <code>null</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public int[] getIntegers(String key) {
    if (key == null)
      return null;
    String[] ss = getStrings(key);
    int[] ii = parseIntegers(ss);
    return ii;
  }
  
  /**
   * Get the integer value stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the value or <code>0</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public int getInteger(String key) {
    if (key == null)
      return 0;
    String s = getString(key);
    int i = parseInteger(s);
    return i;
  }
  
  /**
   * Get the long values stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the values or <code>null</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public long[] getLongs(String key) {
    if (key == null)
      return null;
    String[] ss = getStrings(key);
    long[] ll = parseLongs(ss);
    return ll;
  }
  
  /**
   * Get the long value stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the value or <code>0</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public long getLong(String key) {
    if (key == null)
      return 0;
    String s = getString(key);
    long l = parseLong(s);
    return l;
  }
  
  /**
   * Get the string values stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the values or <code>null</code>, if the key is not found.
   */
  public String[] getStrings(String key) {
    return getStrings(key, false);
  }
  
  /**
   * Get the string values stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @param unquote unquote strings.
   * @return the values or <code>null</code>, if the key is not found.
   */
  public String[] getStrings(String key, boolean unquote) {
    if (key == null)
      return null;
    String[] ss = null;
    String s = null;
    int i1 = 0, i2 = 0, l = key.length();
    while (i2 < l) {
      i2 = key.indexOf('|', i1);
      if (i2 < 0)
        i2 = l;
      s = p.get(key.substring(i1, i2).toLowerCase());
      if (s != null)
        break;
      i1 = i2 + 1;
    }
    if (s == null)  return null;
    ss = tokenize(s);
    if (unquote) {
      for (int i=0; i<ss.length; i++)
        ss[i] = unquote(ss[i]);
    }
    return ss;
  }
  
  /**
   * Get the string value stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the value or <code>null</code>, if the key is not found.
   */ 
  public String getString(String key) {
    return getString(key, false);
  }
  
  /**
   * Get the string value stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @param unquote unquote the string.
   * @return the value or <code>null</code>, if the key is not found.
   */
  public String getString(String key, boolean unquote) {
    if (key == null)
      return null;
    String s = null;
    int i1 = 0, i2 = 0, l = key.length();
    while (i2 < l) {
      i2 = key.indexOf('|', i1);
      if (i2 < 0)
        i2 = l;
      s = p.get(key.substring(i1, i2).toLowerCase());
      if (s != null)
        break;
      i1 = i2 + 1;
    }
    if (unquote && s != null) s = unquote(s);
    return s;
  }

  /**
   * Get the time values stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the values or <code>null</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public int[] getTimes(String key) {
    if (key == null)
      return null;
    String[] ss = getStrings(key, true);
    int[] ii = parseTimes(ss);
    return ii;
  }
  
  /**
   * Get the time value stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the value or <code>0</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public int getTime(String key) {
    if (key == null)
      return 0;
    String s = getString(key, true);
    int i = parseTime(s);
    return i;
  }

  /**
   * Get the date values stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the values or <code>null</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public double[] getDates(String key) {
    if (key == null)
      return null;
    String[] ss = getStrings(key, true);
    double[] dd = parseDates(ss, timezone);
    return dd;
  }
  
  /**
   * Get the date value stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the value or <code>Double.NaN</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public double getDate(String key) {
    if (key == null)
      return 0;
    String s = getString(key, true);
    double d = parseDate(s, timezone);
    return d;
  }
  
  /**
   * Change the key of an entry.
   * @param oldKey the old key.
   * @param newKey the new key.
   * @return the old key or {@code null} if the key doesn't exist.
   */
  public String rename(String oldKey, String newKey) {
    if (oldKey == null)
      return null;
    String s = null;
    String key = null;
    int i1 = 0, i2 = 0, l = oldKey.length();
    while (i2 < l) {
      i2 = oldKey.indexOf('|', i1);
      if (i2 < 0)
        i2 = l;
      key = oldKey.substring(i1, i2).toLowerCase();
      s = p.get(key);
      if (s != null)
        break;
      i1 = i2 + 1;
    }
    if (s == null)  key = null;
    else p.remove(key);
    if (newKey != null)  p.put(newKey.toLowerCase(), s);
    return key;
  }
  
  /**
   * Format the double value and store it under the given key. 
   * @param key the key to be used.
   * @param d the double value.
   * @param format the JAVA format for formatting.
   */
  public void putDouble(String key, double d, String format) {
    String s = String.format(locale, format, d);
    if (locale == Locale.GERMAN)
      s = s.replace('.', ',');
    p.put(key.toLowerCase(), s);
  }

  /**
   * Format the double values and store them under the given key. 
   * @param key the key to be used.
   * @param dd the double values.
   * @param format the JAVA format for formatting.
   */
  public void putDoubles(String key, double[] dd, String format) {
    int n = dd.length;
    StringBuilder sb = new StringBuilder();
    for (int i=0; i<n; i++) {
      if (i > 0) sb.append(' ');
      String s = String.format(locale, format, dd[i]);
      if (locale == Locale.GERMAN)
        s = s.replace('.', ',');
      sb.append(s);
    }
    p.put(key.toLowerCase(), sb.toString());
  }

  /**
   * Format the float value and store it under the given key. 
   * @param key the key to be used.
   * @param f the float value.
   * @param format the JAVA format for formatting.
   */
  public void putFloat(String key, float f, String format) {
    String s = String.format(locale, format, f);
    if (locale == Locale.GERMAN)
      s = s.replace('.', ',');
    p.put(key.toLowerCase(), s);
  }

  /**
   * Format the float values and store them under the given key. 
   * @param key the key to be used.
   * @param ff the float values.
   * @param format the JAVA format for formatting.
   */
  public void putFloats(String key, float[] ff, String format) {
    int n = ff.length;
    StringBuilder sb = new StringBuilder();
    for (int i=0; i<n; i++) {
      if (i > 0) sb.append(' ');
      String s = String.format(locale, format, ff[i]);
      if (locale == Locale.GERMAN)
        s = s.replace('.', ',');
      sb.append(s);
    }
    p.put(key.toLowerCase(), sb.toString());
  }

  /**
   * Format the integer value and store it under the given key. 
   * @param key the key to be used.
   * @param i the integer value.
   * @param format the JAVA format for formatting.
   */
  public void putInteger(String key, int i, String format) {
    String s = String.format(locale, format, i);
    p.put(key.toLowerCase(), s);
  }

  /**
   * Format the integer values and store them under the given key. 
   * @param key the key to be used.
   * @param ii the integer values.
   * @param format the JAVA format for formatting.
   */
  public void putIntegers(String key, int[] ii, String format) {
    int n = ii.length;
    StringBuilder sb = new StringBuilder();
    for (int i=0; i<n; i++) {
      if (i > 0) sb.append(' ');
      sb.append(String.format(locale, format, ii[i]));
    }
    p.put(key.toLowerCase(), sb.toString());
  }

  /**
   * Format the long value and store it under the given key. 
   * @param key the key to be used.
   * @param l the long value.
   * @param format the JAVA format for formatting.
   */
  public void putLong(String key, long l, String format) {
    String s = String.format(locale, format, l);
    p.put(key.toLowerCase(), s);
  }

  /**
   * Format the long values and store them under the given key. 
   * @param key the key to be used.
   * @param ll the long values.
   * @param format the JAVA format for formatting.
   */
  public void putLongs(String key, long[] ll, String format) {
    int n = ll.length;
    StringBuilder sb = new StringBuilder();
    for (int i=0; i<n; i++) {
      if (i > 0) sb.append(' ');
      sb.append(String.format(locale, format, ll[i]));
    }
    p.put(key.toLowerCase(), sb.toString());
  }

  /**
   * Format the time value and store it under the given key. 
   * @param key the key to be used.
   * @param i the time value.
   */
  public void putTime(String key, int i) {
    String s = formatTime(i).trim();
    putString(key, s, true);
  }

  /**
   * Format the time values and store them under the given key. 
   * @param key the key to be used.
   * @param ii the time values.
   */
  public void putTimes(String key, int[] ii) {
    int n = ii.length;
    StringBuilder sb = new StringBuilder();
    for (int i=0; i<n; i++) {
      if (i > 0) sb.append(' ');
      sb.append("\"").append(formatTime(ii[i]).trim()).append("\"");
    }
    p.put(key.toLowerCase(), sb.toString());
  }

  /**
   * Store the string under the given key.
   * @param key the key to be used.
   * @param s the string.
   * @param quoted quote the string.
   */
  public void putString(String key, String s, boolean quoted) {
    if (quoted && s.length()>0 && s.charAt(0) != '"')  s = "\"" + s + "\"";
    p.put(key.toLowerCase(), s);
  }

  /**
   * Store the strings under the given key.
   * @param key the key to be used.
   * @param ss the strings.
   * @param quoted quote the strings.
   */
  public void putStrings(String key, String[] ss, boolean quoted) {
    int n = ss.length;
    StringBuilder sb = new StringBuilder();
    for (int i=0; i<n; i++) {
      if (i > 0) sb.append(' ');
      boolean q = quoted && ss[i].length()>0 && ss[i].charAt(0) != '"';
      if (q) sb.append('"');
      sb.append(ss[i]);
      if (q) sb.append('"');
    }
    p.put(key.toLowerCase(), sb.toString());
  }

  /**
   * Format the date value and store it under the given key. 
   * @param key the key to be used.
   * @param d the date value.
   */
  public void putDate(String key, double d) {
    String s = formatDate(d, timezone);
    p.put(key.toLowerCase(), s);
  }

  /**
   * Format the date values and store them under the given key. 
   * @param key the key to be used.
   * @param dd the date values.
   */
  public void putDates(String key, double[] dd) {
    int n = dd.length;
    StringBuilder sb = new StringBuilder();
    for (int i=0; i<n; i++) {
      if (i > 0) sb.append(' ');
      sb.append(formatDate(dd[i], timezone));
    }
    p.put(key.toLowerCase(), sb.toString());
  }
  
  /**
   * Print a list of all key/value pairs.
   * @param pw the print writer to be used.
   */
  public void print(PrintWriter pw) {
    pw.print(getList());
    pw.flush();
  }
  
  /**
   * Get a list of all key/value pairs.
   * @return the list.
   */
  public String getList() {
    StringBuffer sb = new StringBuffer();
    Set<String> keys = p.keySet();
    for (String key: keys) {
      String value = p.get(key);
      sb.append(key).append(' ').append(value).append('\n');
    }
    return sb.toString();
  }

  public ZonedDate getReferenceDate() {                           //-2010-01-22
    String tmzn =  getString("tmzn|zone", true);
    String rdat = getString("rdat|refdate|refdatum", true);
    return ZonedDate.getZonedDate(tmzn, rdat);
  }
  
  //=========================================================================
  
  private static void test01() {
    String s = " xx\t123 \"abc\" 456\r \"AB\\\"CD\\\"EF\" 789 \n";
    String[] ss = tokenize(s);
    for (int i=0; i<ss.length; i++)
      System.out.printf("%d: %s\n", i, ss[i]);
  }
  
  private static void test02() {
    SimpleDateFormat f = new SimpleDateFormat("HH:mm:ss");
    System.out.println(f.format(new Date()));
    try {
      Date d = f.parse("12:33:44");
      System.out.println("date="+d);
      System.out.println("time="+f.format(d));
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }
  
  private static void test03() {
    try {
      TimeZone z = TimeZone.getTimeZone("GMT+03:00");
      System.out.printf("name=%s, ID=%s\n", z.getDisplayName(), z.getID());
      double d = parseDate("2008-12-01.12:00:00+0100", z);
      Date date = getJDate(d);
      System.out.printf("date=%s\n", date);
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }
  
  /**
   * For testing only.
   * @param args (not used).
   */
  public static void main(String[] args) {
    test03();
  }
  
  //--------------------------------------------------------------//-2010-01-22

  public static class ZonedDate {

    private SimpleDateFormat sdf = new SimpleDateFormat(date_format);
    private long seconds;
    private TimeZone tz;
    private Date date;

    public ZonedDate(TimeZone tz, long seconds) {
      if (tz == null)
        tz = TimeZone.getTimeZone(default_tmzn);
      this.tz = tz;
      this.seconds = seconds;
      date = new Date(seconds*1000);
      sdf.setTimeZone(tz);
    }

    public ZonedDate(String zone, String datum) throws Exception {
      if (datum != null && !datum.contains("."))
        datum += ".00:00:00";
      tz = TimeZone.getTimeZone(zone);
      sdf.setTimeZone(tz);
      date = sdf.parse(datum);
      long ms = date.getTime();
      seconds = ms/1000;
    }

    public long getSeconds() {
      return seconds;
    }

    public TimeZone getZone() {
      return tz;
    }

    public Date getDate() {
      return date;
    }

    @Override
    public String toString() {
      return toString(0);
    }

    public String toString(long t) {
      long ms = (seconds + t)*1000;
      Date d = new Date(ms);
      String s = sdf.format(d) + " " + tz.getID();
      return s;
    }

    public String getDateString(boolean cut) {
      String s = sdf.format(date);
      if (cut && s.endsWith(".00:00:00"))
        s = s.substring(0, s.length()-9);
      return s;
    }

    public String getZoneString() {
      return tz.getID();
    }

    public static ZonedDate getZonedDate(String tmzn, String rdat) {
      ZonedDate zd = null;
      if (rdat == null || rdat.length() == 0) {
        try {
          zd = new ZonedDate(default_tmzn, default_date);
        }
        catch (Exception e) {}
        return zd;
      }
      if (tmzn == null || !tmzn.startsWith("GMT"))
        tmzn = default_tmzn;
      int l = rdat.indexOf("GMT");
      if (l > 0) {
        tmzn = rdat.substring(l);
        rdat = rdat.substring(0, l).trim();
      }
      else {
        l = rdat.length();
        if (l > 5 && ((rdat.charAt(l-5) == '-') || (rdat.charAt(l-5) == '+'))) {
          tmzn = "GMT" + rdat.substring(l-5);
          rdat = rdat.substring(0, l-5).trim();
        }
      }
      try {
        zd = new ZonedDate(tmzn, rdat);
      }
      catch (Exception e) {}
      return zd;
    }

  }
  //-------------------------------------------------------------------------

}

