// ================================================================= IBJarr.java
//
// Utility for handling regular arrays
// ===================================
//
// Copyright 2005-2011 Janicke Consulting, 88662 Überlingen
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of
// the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// History:
//
// 2010-01-25 lj reform() : selection of components
// 2010-01-28 lj Mapping of mapped arrays
// 2010-02-01 lj AbstractArray.setDescriptor(): applying desc.fact
//
// =============================================================================

package de.janicke.ibjutil;

import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.reflect.Array;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.Locale;
import java.util.Vector;

/**
 *
 * <p>Copyright 2005-2011 Janicke Consulting, 88662 Überlingen</p>
 * <p>
 * Container (host) for regular arrays of the same structure (components).
 * Each component is an {@link IBJarr.AbstractArray AbstractArray} and
 * contains data elements of one of the types listed in 
 * {@link IBJdcl IBJdcl}. It is backed up by a regular JAVA array of 1 to 5
 * dimensions. The container class {@code IBJarr} provides the structure
 * information and a header of type {@link IBJhdr} for all components.
 * </p>
 * <p><b>Example for using <code>IBJarr</code>:</b></p>
 * <p>
 * First the container is created, setting the name, the number of dimensions
 * and the number of index values. Then the start values of the indices and
 * the time zone are set and some information is put into the header:
 * <pre>
      IBJarr arr = new IBJarr("test", 3, 4);
      arr.setFirstIndex(1, 1);
      arr.setTimeZone(null);
      IBJhdr hdr = arr.getHeader();
      hdr.putString("title", "A simple example", true);
      hdr.putDate("created", IBJhdr.getDate(new Date())); </pre>
 * Then two float arrays are created with names <i>z0</i> and <i>d0</i>. The values 
 * of <i>z0</i> are set using the provided setter method:
 * <pre>
      arr.createArray("z0%5.2f");
      FloatArray fa_z0 = (FloatArray)arr.getArray("z0");
      arr.createArray("d0%5.2f");
      FloatArray fa_d0 = (FloatArray)arr.getArray("d0");
      for (int i=1; i<=3; i++) {
        for (int j=1; j<=4; j++) {
          fa_z0.set(0.1f*i, i, j);
        }
      } </pre>
 * The values of <i>d0</i> are set by working immediately with the genuine
 * JAVA arrays backing up the FloatArrays:
 * <pre>
      float[][] z0 = (float[][])fa_z0.getData();
      float[][] d0 = (float[][])fa_d0.getData();
      for (int i=0; i<3; i++)
        for (int j=0; j<4; j++)
          d0[i][j] = 6*z0[i][j]; </pre>
 * Finally a print writer is created and structure information, the header and
 * the data are printed:
 * <pre>
      PrintWriter pw = new PrintWriter(new OutputStreamWriter(System.out));
      arr.printInfo(pw);
      pw.println("\nHeader:");
      hdr.print(pw);
      arr.printAll("\nThe data of " + hdr.getString("title", false), pw); </pre>
 *
 * @author Lutz Janicke, Janicke Consulting, Dunum
 * @version 2010-02-01
 */

public class IBJarr implements IBJdcl {
  /** Print check output to {@code System.err}. */
  public static boolean CHECK = false;
  /** The components of this container. */
  public Vector<AbstractArray> arrays;
  private Charset charset = Charset.defaultCharset();
  private ByteOrder byteorder = ByteOrder.nativeOrder();
  private float factor = 1;
  private String host_name;
  private Structure structure;
  private Mapping mapping;
  private IBJhdr header;

  private IBJarr(String name) {
    host_name = name;
  }

  /**
   * Create a new container with a given structure.
   * @param name the name given to this container.
   * @param ii the number of index values for each dimension.
   * @throws Exception
   */
  public IBJarr(String name, int... ii) throws Exception {
    host_name = name;
    structure = new Structure(ii);
    header = new IBJhdr();
    arrays = new Vector<AbstractArray>();
  }

  /**
   * Get the name of this array container.
   * @return the name of this array container.
   */
  public String getName() {
    return host_name;
  }
  
  /**
   * Get the header (key/value definitions) of this array container.
   * @return the header.
   */
  public IBJhdr getHeader() {
    return header;
  }

  /**
   * Set the header of this array container. A new header is created and filled
   * with the content of the given header.
   * @param hdr the header provided.
   */
  public void setHeader(IBJhdr hdr) {
    header = new IBJhdr(hdr);
  }

  /**
   * Set the lower bound of the index range for each dimension.
   * @param ii the lower bounds.
   * @throws Exception
   */
  public void setFirstIndex(int... ii) throws Exception {
//    if (arrays.size() > 0)  throw new Exception("invalid operation"); 2008-11-13
    structure.setFirstIndex(ii);
  }

  /**
   * Get the structure of the component array in this array container.
   * @return the structure.
   */
  public Structure getStructure() {
    return structure;
  }

  /**
   * Get the mapping of the component array in this array container.
   * @return the mapping or null if not mapped.
   */
  public Mapping getMapping() {
    return mapping;
  }

  /**
   * Check whether the structure of a JAVA array is compatible with the
   * structure used for the component arrays in this array container.
   * @param array a JAVA array
   * @return is compatible.
   */
  public boolean checkStructure(Object array) {
    Vector<Integer> v = new Vector<Integer>();
    for (int i : structure.length)
      v.add(i);
    return checkStructure(array, v);
  }

  private boolean checkStructure(Object array, Vector<Integer> pv) {  //-2008-11-13
    boolean is_compatible = true;
    try {
      int n = Array.getLength(array);
      if (n != pv.get(0)) return false;
      if (pv.size() > 1) {
        Vector<Integer> v = new Vector<Integer>();                //-2008-11-13
        for (int i=1; i<pv.size(); i++)                           //-2008-11-13
          v.add(pv.get(i));                                       //-2008-11-13
        for (int i = 0; i < n; i++) {
          Object b = Array.get(array, i);
          is_compatible = checkStructure(b, v);
          if (!is_compatible) break;
        }
      }
    }
    catch (Exception e) {
      e.printStackTrace();                                        //-2008-11-13
      is_compatible = false;
    }
    return is_compatible;
  }

  /**
   * Create a component array of the the given type. The name must be unique 
   * within this array container. The component created is appended to the vector
   * of components.
   * @param format the format defining the name, element type and representation.
   * @return the component array created.
   * @throws Exception
   */
  public AbstractArray createArray(String format) throws Exception {
    Descriptor dsc = new Descriptor(format);
    return createArray(dsc, null);
  }

  /**
   * Create a component array backing up its data from the given JAVA array.
   * The name must be unique 
   * within this array container. The component created is appended to the vector
   * of components.
   * @param format the format defining the name, element type and representation.
   * @param array the JAVA array providing the data.
   * @return the component array created.
   * @throws Exception
   */
  public AbstractArray createArray(String format, Object array) throws Exception {
    Descriptor dsc = new Descriptor(format);
    return createArray(dsc, array);
  }

  private AbstractArray createArray(Descriptor desc, Object array) throws Exception {
    if (array != null) {
      if (!checkStructure(array))
        throw new Exception("invalid structure of data array");
    }
    for (int i = 0; i < arrays.size(); i++) {
      if (desc.name.equals(arrays.get(i).getName()))
        throw new Exception("duplicate name");
    }
    AbstractArray absarr = null;
    switch (desc.type) {
      case BYTE:
        absarr = new ByteArray(array);
        break;
      case SHORT:
        absarr = new ShortArray(array);
        break;
      case INTEGER:
        absarr = new IntegerArray(array);
        break;
      case LONG:
        absarr = new LongArray(array);
        break;
      case FLOAT:
        absarr = new FloatArray(array);
        break;
      case DOUBLE:
        absarr = new DoubleArray(array);
        break;
      case STRING:
        absarr = new StringArray(array);
        break;
      case TIME:
        absarr = new TimeArray(array);
        break;
      case DATE:
        absarr = new DateArray(array);
        break;
      default:
        throw new Exception("type "+desc.type+" not implemented");
    }
    absarr.init(desc);
    arrays.add(absarr);
    if (CHECK)
      System.err.println("array " + desc.name + " inserted at "
          + (arrays.size() - 1));
    return absarr;
  }

  /**
   * Get the number of component arrays hosted by this array container.
   * @return the number of components.
   */
  public int getSize() {
    return arrays.size();
  }

  /**
   * Get a component array by sequence number.
   * @param index the sequence number (>=0)
   * @return the component array
   * @throws Exception
   */
  public AbstractArray getArray(int index) throws Exception {
    if (index < 0 || index >= arrays.size())
      throw new Exception(String.format("index value %d not in range [%d,%d]", 
          index, 0, arrays.size()-1));
    return arrays.get(index);
  }

  /**
   * Get a component array by name.
   * @param name the name of the component.
   * @return the component array.
   */
  public AbstractArray getArray(String name) {
    AbstractArray a = null;
    for (int i = 0; i < arrays.size(); i++) {
      a = arrays.get(i);
      if (name.equals(a.getName())) break;
      a = null;
    }
    return a;
  }

  /**
   * Set a new data format for a component array. The name must be unique
   * within the array container, the data type must not be changed.
   * @param name the name of the component array.
   * @param format the new format (including name).
   * @throws Exception
   */
  public void setFormat(String name, String format) throws Exception {
    AbstractArray aa = getArray(name);
    if (aa == null) throw new Exception("array not found");
    Descriptor d = new Descriptor(format);
    AbstractArray bb = getArray(d.name);
    if (bb != null && bb != aa)
      throw new Exception("duplicate name in format specification");
    if (d.type != aa.desc.type)
      throw new Exception("invalid data type in format specification");
    aa.init(d);
  }

  /**
   * Change the name of a component array. The name must be unique within
   * the array container.
   * @param oldName the old name.
   * @param newName the new name.
   * @throws Exception
   */
  public void setName(String oldName, String newName) throws Exception {
    if (oldName.equals(newName)) return;
    AbstractArray aa = getArray(oldName);
    if (aa == null) throw new Exception("array not found");
    AbstractArray bb = getArray(newName);
    if (bb != null && bb != aa) throw new Exception("duplicate name");
    aa.desc.name = newName;
  }

  /**
   * Checks, whether the component arrays own the data or whether they refer
   * to data of component arrays in another array container.
   * @return true if a mapping exists.
   */
  public boolean isMapped() {
    return (mapping != null);
  }

  /**
   * Get an array container with component arrays, whose data are mapped on 
   * the component arrays of this container. The mapping is given by a 
   * string defining a new sequence of index values and comprises
   * reordering and/or selection of data elements. 
   * <br><br>
   * Reordering is specified by a comma separated list of index names (i,j,...)
   * corresponding to the dimensions of the array. The rightmost positioned index
   * is varied most quickly. Therefore, for a 3-dimensional array A(i,j,k) the sequence 
   * {@code i,j,k} specifies the natural ordering in C
   * and {@code k,j,i} is the natural ordering in FORTRAN.
   * <br>
   * If a minus symbol is appended to a letter, then this index runs backwards. Thus
   * {@code j-,i+} is the geographical correct ordering of a 2-dimensional array
   * A(i,j) in a formatted output
   * if the index i corresponds to the x-direction and j to the y-direction.
   * <br>
   * A selection is either a reduction in the number of dimensions by specifying
   * an index value (e.g. {@code k=1}) or a new range of index 
   * values (e.g. {@code k=1..10}). Optionally a new starting value for the reduced
   * index range may be specified by appending this value separated by a slash 
   * ({@code /}). Therefore, a selection of 5x5 elements in a horizontal layer
   * of the array A(i,j,k) may be specified by {@code j=10..5/1,i=5..10/1,k=3}.
   * <br>
   * @param name new name of the new container.
   * @param selection specification of the mapping.
   * @return the new array container. 
   * @throws Exception
   */
  public IBJarr getSelected(String name, String selection) throws Exception {
    IBJarr arr = new IBJarr(name);
    if (selection == null) {
      arr.mapping = mapping;
      arr.structure = structure;
    }
    else {
      arr.mapping = new Mapping(structure, selection, mapping);
      arr.structure = arr.mapping.src;
    }
    arr.header = new IBJhdr(header);
    arr.header.rename("sequ", null);
    arr.header.rename("lowb", null);
    arr.header.rename("hghb", null);
    arr.header.rename("dims", null);
    arr.arrays = new Vector<AbstractArray>();
    for (int i = 0; i < arrays.size(); i++) {
      AbstractArray aa = arrays.get(i);
      AbstractArray bb = null;                                    //-2008-11-14
      switch (aa.desc.type) {
        case BYTE:
          arr.arrays.add(bb = arr.new ByteArray((ByteArray) aa));       //-2008-11-14
          break;
        case SHORT:
          arr.arrays.add(bb = arr.new ShortArray((ShortArray) aa));     //-2008-11-14
          break;
        case INTEGER:
          arr.arrays.add(bb = arr.new IntegerArray((IntegerArray) aa)); //-2008-11-14
          break;
        case LONG:
          arr.arrays.add(bb = arr.new LongArray((LongArray) aa));       //-2008-11-14
          break;
        case FLOAT:
          arr.arrays.add(bb = arr.new FloatArray((FloatArray) aa));     //-2008-11-14
          break;
        case DOUBLE:
          arr.arrays.add(bb = arr.new DoubleArray((DoubleArray) aa));   //-2008-11-14
          break;
        case STRING:
          arr.arrays.add(bb = arr.new StringArray((StringArray) aa));   //-2008-11-14
          break;
        case TIME:
          arr.arrays.add(bb = arr.new TimeArray((TimeArray) aa));       //-2008-11-14
          break;
        case DATE:
          arr.arrays.add(bb = arr.new DateArray((DateArray) aa));       //-2008-11-14
          break;
        default:
          throw new Exception("unknown type");
      }
      if (bb != null)  bb.fact = aa.fact;                         //-2008-11-14
    }
    return arr;
  }

  /**
   * Get a deep copy of this array container.
   * @param name the new name.
   * @return the copy
   * @throws Exception
   */
  public IBJarr getCopy(String name) throws Exception {
    IBJarr arr = new IBJarr(name, structure.length);
    arr.setFirstIndex(structure.first);
    arr.header = new IBJhdr(header);
    arr.arrays = new Vector<AbstractArray>();
    for (int i = 0; i < arrays.size(); i++) {
      AbstractArray aa = arrays.get(i);
      AbstractArray cc = aa.getCopy(arr);
      arr.arrays.add(cc);
    }
    return arr;
  }

  /**
   * Print structure and element information of this array container.
   * @param pw the print writer to be used.
   */
  public void printInfo(PrintWriter pw) {
    if (pw == null)
      return;
    pw.printf("IBJarr %s", host_name);
    if (isMapped()) {
      pw.printf(" (mapped)");
    }
    pw.println();
    structure.print("Structure:", pw);
    pw.printf("Components:\n");
    for (int i = 0; i < arrays.size(); i++) {
      AbstractArray aa = arrays.get(i);
      pw.printf("%2d: %8s %12s %8s %2d %s\n", i + 1, aa.desc.name, aa.desc.cfrm,
          aa.desc.jfrm, aa.desc.length, aa.desc.type);
    }
  }
  
  /**
   * Get structure and element information of this array container.
   * @return a string holding the information.
   */
  public String listInfo() {
    StringWriter sw = new StringWriter();
    PrintWriter pw = new PrintWriter(sw);
    printInfo(pw);
    return sw.toString();
  }
  
  /**
   * Print all data of the component arrays of this array container.
   * @param title an optional title line.
   * @param pw the print writer to be used.
   * @throws Exception
   */
  public void printAll(String title, PrintWriter pw) throws Exception {
    printAll(title, pw, ' ');
  }

  /**
   * Print all data of the component arrays of this array container.
   * @param title an optional title line.
   * @param pw the print writer to be used.
   * @param sep character used for separating items
   * @throws Exception
   */
  public void printAll(String title, PrintWriter pw, char sep) throws Exception {
    if (pw == null)
      return;
    if (title != null) pw.println(title);
    int ns = structure.dims;
    int[] ii = (int[]) structure.first.clone();
    int na = arrays.size();
    int i = 0;
    while (i >= 0) {
      for (int k = 0; k < na; k++) {
        AbstractArray aa = arrays.get(k);
        pw.print(sep);
        if (aa.getType() == DataType.STRING)
          pw.printf("\"%s\"", aa.format(ii));
        else
          pw.print(aa.format(ii));
      }
      if (na > 1) pw.println();
      for (i = ns - 1; i >= 0; i--) {
        ii[i]++;
        if (ii[i] < structure.first[i] + structure.length[i])
          break;
        else {
          ii[i] = structure.first[i];
          pw.println();
        }
      }
    }
    pw.flush();
  }

  /**
   * Set the time zone of the header (see {@link IBJhdr#setTimeZone(String)
   * IBJhdr.setTimeZone()}). The default time 
   * zone is used if the argument is {@code null}.
   * @param s the ID of the time zone as GMT[+|-]mmss.
   */
  public void setTimeZone(String s) {
    header.setTimeZone(s);
  }

  /**
   * Get the time zone of the header as string.
   * @return the ID of the time zone.
   */
  public String getTimeZone() {
    return header.getTimeZone().getID();
  }

  /**
   * Set the byte order for storage in binary files.
   * @param lsbf least significant byte first.
   */
  public void setLSBF(boolean lsbf) {
    byteorder = (lsbf) ? ByteOrder.LITTLE_ENDIAN : ByteOrder.BIG_ENDIAN;
    int na = arrays.size();
    for (int i = 0; i < na; i++) {
      AbstractArray aa = arrays.get(i);
      aa.buffer.order(byteorder);
    }
  }

  /**
   * Get the byte order used for storage in binary files.
   * @return least significant byte first.
   */
  public boolean isLSBF() {
    return (byteorder == ByteOrder.LITTLE_ENDIAN);
  }

  /**
   * Set the locale to be used in formatting data (see 
   * {@link IBJhdr#setLocale(String) IBJhdr.setLocale()}).
   * @param locale "C" or "german".
   */
  public void setLocale(String locale) {
    header.setLocale(locale);
  }

  /**
   * Get the locale used for formatting data.
   * @return the current locale.
   */
  public Locale getLocale() {
    return header.getLocale();
  }

  /**
   * Set the character set to be used in converting characters to bytes.
   * @param chsn the name of the character set.
   */
  public void setCharset(String chsn) {
    charset = Charset.forName(chsn);
  }

  /**
   * Get the character set used in converting characters to bytes.
   * @return the name of the character set.
   */
  public String getCharset() {
    return charset.name();
  }

  /**
   * Set the factor to be used for scaling float numbers for formatted output.
   * @param f the scaling factor.
   */
  public void setFactor(float f) {
    factor = f;
    int na = arrays.size();
    for (int i = 0; i < na; i++) {
      AbstractArray aa = arrays.get(i);
      if (Float.isNaN(aa.desc.fact)) aa.fact = f;
    }
  }

  /**
   * Get the factor used for scaling float numbers for formatted output.
   * @return the scaling factor.
   */
  public float getFactor() {
    return factor;
  }

  /**
   * Select and/or reorder components of an array. The components are
   * ordered and selected corresponding to their names in the parameter
   * <code>form</code>. If the field length of an format item is  0 then
   * the corresponding component is dropped. After splitting the combined
   * format the work is delegated to reform(String[], boolean).
   *
   * @param form  the new combined format for the created array
   * @param force_new create a new array, even if the sequence of
   * components is not changed.
   * @return the array created
   * @throws Exception
   */
  public IBJarr reform(String form, boolean force_new) throws Exception {
    String[] forms = Descriptor.expandFormat(form);
    return reform(forms, force_new);
  }

  /**
   * Select and/or reorder components of an array. The components are
   * ordered and selected corresponding to their names in the parameter
   * <code>forms</code>. If the field length of an format item is  0 then
   * the corresponding component is dropped.
   *
   * @param forms  the new format vector for the created array
   * @param force_new create a new array, even if the sequence of
   * components is not changed.
   * @return the array created
   * @throws Exception
   */
  public IBJarr reform(String[] forms, boolean force_new) throws Exception {
    ArrayList<Descriptor> list = new ArrayList<Descriptor>();
    HashSet<Integer> setk = new HashSet<Integer>();
    int nk = getSize();
    int i, k;
    for (i=0; i<forms.length; i++) {
      Descriptor dsc = new Descriptor(forms[i]);
      if (dsc.width.equals("0"))
        continue;
      list.add(dsc);
    }
    int nd = list.size();
    int[] kk = new int[nd];
    if (nd == nk && !force_new) {
      for (i=0; i<nk; i++) {
        Descriptor dsc = list.get(i);
        Descriptor d = arrays.get(i).getDescriptor();
        if (!dsc.name.equalsIgnoreCase(d.name))
          break;
        if (!dsc.format.equals(d.format))
          break;
      }
      if (i >= nk)
        return this;
    }
    for (i=0; i<nd; i++) {
      Descriptor dsc = list.get(i);
      for (k=0; k<nk; k++) {
        Descriptor d = arrays.get(k).getDescriptor();
        if (dsc.name.equalsIgnoreCase(d.name) && dsc.type.equals(d.type))
          break;
      }
      if (k >= nk)
        throw new Exception("No match found for format \"" + forms[i] + "\"");
      if (setk.contains(k))
        throw new Exception("Duplicate match for format \"" + forms[i] + "\"");
      kk[i] = k;
      setk.add(k);
    }
    //
    IBJarr arr = new IBJarr(host_name, structure.length);
    arr.setFirstIndex(structure.first);
    arr.header = new IBJhdr(header);
    arr.arrays = new Vector<AbstractArray>();
    String[] new_forms = new String[nd];
    for (i=0; i<nd; i++) {
      Descriptor dsc = list.get(i);
      k = kk[i];
      AbstractArray aa = arrays.get(k);
      aa.setDescriptor(dsc);
      if (!Float.isNaN(dsc.fact)) aa.fact = dsc.fact;             //-2010-02-01
      else aa.fact = arr.factor;
      arr.arrays.add(aa);
      new_forms[i] = dsc.format;
    }
    arr.header.putStrings("form", new_forms, true);
    arr.header.rename("size", null);
    //
    return arr;
  }

  // ============================================================== Structure
  /**
   * Information about the structure of component arrays.
   */
  public static class Structure {
    private int dims;
    private int[] first;
    private int[] last;
    private int[] length;

    private Structure(int dims) {
      this.dims = dims;
      first = new int[dims];
      last = new int[dims];
      length = new int[dims];
    }

    private Structure(int... ii) {
      dims = ii.length;
      first = new int[dims];
      last = new int[dims];
      length = new int[dims];
      for (int i = 0; i < dims; i++) {
        length[i] = ii[i];
        last[i] = ii[i] - 1;
      }
    }

    /**
     * Compares this structure with another structure. They are compatible if
     * they coincide with respect to number of dimensions and length of index
     * ranges.
     * @param s structure to be compared.
     * @return is compatible.
     */
    public boolean equals(Structure s) {
      if (s == null) return false;
      if (s.dims != dims) return false;
      for (int i = 0; i < dims; i++) {
        if (s.length[i] != length[i]) return false;
      }
      return true;
    }

    /**
     * Set the lowest index value for each dimension.
     * @param ii lowest index values.
     * @throws Exception
     */
    private void setFirstIndex(int... ii) throws Exception {
      if (ii.length != dims) throw new Exception("invalid length");
      for (int i = 0; i < dims; i++) {
        first[i] = ii[i];
        last[i] = ii[i] + length[i] - 1;
      }
    }

    /**
     * Get the number of dimensions.
     * @return number of dimensions.
     */
    public int getDims() {
      return dims;
    }

    /**
     * Get the lowest index value for each dimension.
     * @return lowest index values.
     */
    public int[] getFirstIndex() {
      return (int[]) first.clone();
    }

    /**
     * Get the highest index value for each dimension.
     * @return highest index values.
     */
    public int[] getLastIndex() {
      return (int[]) last.clone();
    }

    /**
     * Get the length of the index range (number of index values) for each
     * dimension.
     * @return lengths of index ranges.
     */
    public int[] getLength() {
      return (int[]) length.clone();
    }

    /**
     * Print the structure (first and last index value of each dimension).
     * @param title optional title line.
     * @param pw the print writer to be used.
     */
    public void print(String title, PrintWriter pw) {
      pw.printf("%s\n", title);
      for (int i = 0; i < dims; i++) {
        pw.printf("%2d: %3d ... %3d\n", i + 1, first[i], last[i]);
      }
      pw.flush();
    }
  }

  // ================================================================= Mapping
  public static class Mapping {
    public Structure src;  // the structure of the mapped array
    public Structure dst;  // the structure of the array containing the data
    public int[][] mat;                                           //-2010-01-28
    int[] dspl;                                                   //-2010-01-28

    private Mapping(Structure dst, String seq, Mapping mapping) throws Exception {
      this.dst = dst;
      int[][] ii = parse(seq);
      int nd = dst.dims;
      int ns = 0;
      for (int n = 0; n < nd; n++) {
        if (ii[n][1] != 0) ns++;
      }
      src = new Structure(ns);
      int[] shift = new int[nd];
      int[][] m2 = new int[nd][ns];
      ns = 0;
      for (int n = 0; n < nd; n++) {
        int m = ii[n][0];
        if (ii[n][1] != 0) {
          m2[m][ns] = ii[n][1];
          src.first[ns] = ii[n][4];
          src.length[ns] = 1 + ((ii[n][1] > 0) ? ii[n][3] - ii[n][2] : ii[n][2]
              - ii[n][3]);
          shift[m] = ii[n][2];
          ns++;
        }
        else {
          shift[m] = ii[n][2];
        }
      }
      for (int n = 0; n < ns; n++)
        src.last[n] = src.first[n] + src.length[n] - 1;
      //-------------------------------------------------------------2010-01-28
      int[] d1 = new int[nd];
      for (int n=0; n<nd; n++) {
        d1[n] = shift[n];
        for (int i=0; i<ns; i++)
          d1[n] -= m2[n][i]*src.first[i];
      }
      if (mapping != null) {
        this.dst = mapping.dst;
        int[][] m1 = mapping.mat;
        int[] d0 = mapping.dspl;
        int n0 = m1.length;
        int n1 = m2.length;
        int n2 = ns;
        int[][] ms = new int[n0][n2];
        for (int i0=0; i0<n0; i0++) {
          for (int i2=0; i2<n2; i2++) {
            int isum = 0;
            for (int i1=0; i1<n1; i1++)
              isum += m1[i0][i1]*m2[i1][i2];
            ms[i0][i2] = isum;
          }
        }
        int[] ds = new int[n0];
        for (int i0=0; i0<n0; i0++) {
          ds[i0] = d0[i0];
          for (int i1=0; i1<n1; i1++)
            ds[i0] += m1[i0][i1]*d1[i1];
        }
        dspl = ds;
        mat = ms;
      }
      else {
        dspl = d1;
        mat = m2;
      }
      //----------------------------------------------------------------------
    }

    private void print(String title, PrintWriter pw) {            //-2010-01-28
      int nd = dspl.length;
      int ns = src.first.length;
      pw.printf("%s\n", title);
      pw.printf("first\n");
      for (int j = 0; j < ns; j++) {
        pw.printf(" %3d  ", src.first[j]);
        for (int i = 0; i < nd; i++)
          pw.printf(" %3d", mat[i][j]);
        if (j == 0) {
          pw.printf("   ");
        }
        pw.printf("\n");
      }
      pw.printf("\ndspl: ");
      for (int i = 0; i < nd; i++)
        pw.printf(" %3d", dspl[i]);
      pw.printf("\n\n");
      pw.flush();
    }

    private void print(String title, int[][] mm, int[] dd, int[] ff) {  //-2010-01-28
      PrintWriter pw = new PrintWriter(System.out);
      int nd = dd.length;
      int ns = ff.length;
      pw.printf("%s\n", title);
      pw.printf("first\n");
      for (int j = 0; j < ns; j++) {
        pw.printf(" %3d  ", ff[j]);
        for (int i = 0; i < nd; i++)
          pw.printf(" %3d", mm[i][j]);
        if (j == 0) {
          pw.printf("   ");
        }
        pw.printf("\n");
      }
      pw.printf("\ndspl: ");
      for (int i = 0; i < nd; i++)
        pw.printf(" %3d", dd[i]);
      pw.printf("\n\n");
      pw.flush();
    }

    public int[][] parse(String seq) throws Exception {
      int nd = dst.dims;
      boolean[] defined = new boolean[nd];
      seq = IBJhdr.unquote(seq);                                  //-2010-05-10
      String[] tt = seq.split("[,;:]");
      if (CHECK) for (int i = 0; i < tt.length; i++)
        System.err.println(i + ": " + tt[i]);
      if (tt.length != nd)
        throw new Exception("mapping has invalid dimension");
      int[][] ii = new int[nd][5];
      int nvi = Character.getNumericValue('i');
      for (int n = 0; n < nd; n++) {
        String t = tt[n];
        if (t.length() < 1)
          throw new Exception("invalid token(" + (n + 1) + ")");
        int m = Character.getNumericValue(t.charAt(0)) - nvi;
        if (m < 0 || m >= nd)
          throw new Exception("invalid index character \"" + t.charAt(0)
              + "\" (dims=" + nd + ")");
        if (defined[m])
          throw new Exception("duplicate specification for index \""
              + t.charAt(0) + "\"");
        defined[m] = true;
        ii[n][0] = m;
        if (t.length() == 1 || t.charAt(1) == '+') {
          ii[n][1] = 1;
          ii[n][2] = dst.first[m];
          ii[n][3] = ii[n][2] + dst.length[m] - 1;
          ii[n][4] = ii[n][2];
        }
        else if (t.charAt(1) == '-') {
          ii[n][1] = -1;
          ii[n][3] = dst.first[m];
          ii[n][2] = ii[n][3] + dst.length[m] - 1;
          ii[n][4] = ii[n][3];
        }
        else if (t.charAt(1) == '=') {
          String s = t.substring(2);
          int k = s.indexOf("..");
          try {
            if (k < 0) {
              ii[n][2] = Integer.parseInt(s);
              ii[n][4] = ii[n][2];
            }
            else {
              int l = s.indexOf('/', k + 2);
              ii[n][2] = Integer.parseInt(s.substring(0, k));
              if (l < 0) {
                ii[n][3] = Integer.parseInt(s.substring(k + 2));
                ii[n][4] = (ii[n][3] >= ii[n][2]) ? ii[n][2] : ii[n][3];
              }
              else {
                ii[n][3] = Integer.parseInt(s.substring(k + 2, l));
                ii[n][4] = Integer.parseInt(s.substring(l + 1));
              }
              ii[n][1] = (ii[n][3] >= ii[n][2]) ? 1 : -1;
            }
          }
          catch (Exception e) {
            e.printStackTrace();
            throw new Exception("parsing error for token \"" + t + "\"");
          }
        }
        else throw new Exception("invalid mapping in token \"" + t + "\"");
        if (CHECK)
          System.err.printf("%d: %3d %3d %3d %3d %3d\n", n, ii[n][0], ii[n][1],
              ii[n][2], ii[n][3], ii[n][4]);
      }
      return ii;
    }

    /**
     * Get the index vector jj referring to the original data from the
     * index vector ii referring to the mapped array.
     *
     *   jj = map * (ii - first) + shift
     *
     * Simplified (2010-01-28) to
     *
     *   jj = map * ii + dspl
     *
     * @param ii index vector referring to original data
     * @return index vector referring to the mapped data
     * @throws Exception
     */
    public int[] map(int... ii) throws Exception {                //-2010-01-28
      int ni = src.dims;          // dimension of mapped data
      int nj = dst.dims;          // dimension of original data
      if (ii.length != ni)
        throw new Exception("improper dimension");
      int[] jj = new int[nj];
      for (int j = 0; j < nj; j++) {
        for (int i = 0; i < ni; i++) {
          jj[j] += mat[j][i]*ii[i];
        }
        jj[j] += dspl[j];
      }
      return jj;
    }
  }

  // ============================================================ AbstractArray
  /**
   * Base class for all component arrays.
   */
  public abstract class AbstractArray {
    protected Object data;
    protected String valdef;
    protected Descriptor desc;
    protected ByteBuffer buffer;
    protected byte[] bytes;
    protected float fact = 1;

    /**
     * Provides a deep copy of this component array. The copy is not added
     * to the referenced array container.
     * @param arr the array container which the copy refers to.
     * @return deep copy of the component array.
     * @throws Exception
     */
    public abstract AbstractArray getCopy(IBJarr arr) throws Exception;

    /**
     * Get the formatted value of a data element. See {@link IBJarr.Descriptor
     * Descriptor} for formatting options.
     * @param ii the index values of the data element.
     * @return  the formatted value.
     * @throws Exception
     */
    public abstract String format(int... ii) throws Exception;

    /**
     * Parse a string and set the value of a data element.
     * @param value the string to be parsed.
     * @param ii the index values of the data element.
     * @throws Exception
     */
    public abstract void parse(String value, int... ii) throws Exception;

    /**
     * Get the value of a data element from the bytes in an internal buffer.
     * @param ii the index values of the data element.
     * @throws Exception
     */
    protected abstract void fromBuffer(int... ii) throws Exception;

    /**
     * Put the byte representation of the value of an element into an
     * internal buffer.
     * @param ii the index values of the data element.
     * @throws Exception
     */
    protected abstract void toBuffer(int... ii) throws Exception;

    /**
     * Get the data type (see {@link IBJdcl IBJdcl}) of the data elements in 
     * this component array.
     * @return the data type.
     */
    public DataType getType() {
      return desc.type;
    }

    /**
     * Get the name of this component array.
     * @return the name.
     */
    public String getName() {
      return desc.name;
    }

    /**
     * Get the format used to define the element type of this component array.
     * @return the format string.
     */
    public String getFormat() {
      return desc.format;
    }

    /**
     * Get the name of the array container hosting this component array.
     * @return the host name.
     */
    public String getHost() {
      return host_name;
    }

    /**
     * Get the JAVA array backing up the data elements of this component
     * array. If the component array is mapped 
     * (see {@link IBJarr#getMapped(String, String) getMapped()}) a copy of
     * the selected data elements is returned.
     * @return data array.
     * @throws Exception
     */
    public Object getData() throws Exception {
      if (!isMapped())
        return data;
      IBJarr arr = new IBJarr(null, structure.length);
      AbstractArray aa = getCopy(arr);
      return aa.data;
    }

    /**
     * Get the structure of this component array.
     * @return the structure.
     */
    public Structure getStructure() {
      return structure;
    }

    /**
     * Get the descriptor defining the properties of the data elements.
     * @return the descriptor.
     */
    public Descriptor getDescriptor() {
      return desc.getCopy();
    }
    
    /**
     * Redefine the descriptor for the data elements. 
     * @param dsc the new descriptor (a copy is used).
     * @throws Exception if the new and the old descriptor refer to 
     * different data types.
     */
    public void setDescriptor(Descriptor dsc) throws Exception {
      if (dsc == null)
        return;
      if (desc == null)
        throw new Exception("no old descriptor");
      if (desc.type != dsc.type)
        throw new Exception("invalid data type");
      desc = dsc.getCopy();
      if (!Float.isNaN(dsc.fact)) fact = dsc.fact;                //-2010-02-01
      else fact = 1;
    }
    
    /**
     * Get the value definition parameter of the data elements.
     * @return the valdef parameter.
     */
    public String getValdef() {
      return valdef;
    }

    /**
     * Set the value definition parameter of the data elements (one of the
     * letters P,V,X,Y,Z).
     * @param valdef the valdef parameter.
     */
    public void setValdef(String valdef) {
      this.valdef = valdef;
    }
    
    /**
     * Checks, whether the component array owns the data or whether it refers
     * to the data of another component array.
     * @return true if it is mapped.
     */
    public boolean isMapped() {
      return (mapping != null);
    }

    /**
     * Print all data elements in natural sequence using the formatting
     * information saved in the descriptor.
     * @param title an optional title line.
     * @param pw the print writer to be used.
     * @throws Exception
     */
    public void print(String title, PrintWriter pw) throws Exception {
      if (title != null) pw.println(title);
      int ns = structure.dims;
      int[] ii = (int[]) structure.first.clone();
      int i = 0;
      while (i >= 0) {
        pw.print(' ');
        pw.print(format(ii));
        for (i = ns - 1; i >= 0; i--) {
          ii[i]++;
          if (ii[i] <= structure.last[i]) break;
          else {
            ii[i] = structure.first[i];
            pw.println();
          }
        }
      }
      pw.flush();
    }

    /**
     * Initialize the component array. A copy of the descriptor provided is 
     * saved and the byte buffer is created.
     * @param desc the descriptor for this component array.
     */
    protected void init(Descriptor desc) {
      this.desc = desc.getCopy();
      int buflen = desc.length;
      buffer = ByteBuffer.allocate(buflen);
      buffer.order(byteorder);
      bytes = buffer.array();
      if (!Float.isNaN(desc.fact)) fact = desc.fact;
      else fact = factor;
    }
  }

  // ============================================================== ByteArray
  /**
   * A component array containing <i>BYTE</i> values.
   */
  public class ByteArray extends AbstractArray {
    byte[] f1;
    byte[][] f2;
    byte[][][] f3;
    byte[][][][] f4;
    byte[][][][][] f5;

    private ByteArray(Object a) throws Exception {
      int[] ii = structure.getLength();
      switch (ii.length) {
        case 1:
          f1 = (a == null) ? new byte[ii[0]] : (byte[]) a;
          data = f1;
          break;
        case 2:
          f2 = (a == null) ? new byte[ii[0]][ii[1]] : (byte[][]) a;
          data = f2;
          break;
        case 3:
          f3 = (a == null) ? new byte[ii[0]][ii[1]][ii[2]] : (byte[][][]) a;
          data = f3;
          break;
        case 4:
          f4 = (a == null) ? new byte[ii[0]][ii[1]][ii[2]][ii[3]]
              : (byte[][][][]) a;
          data = f4;
          break;
        case 5:
          f5 = (a == null) ? new byte[ii[0]][ii[1]][ii[2]][ii[3]][ii[4]]
              : (byte[][][][][]) a;
          data = f5;
          break;
        default:
          throw new Exception("invalid dimension");
      }
    }

    private ByteArray(ByteArray src) {
      desc = src.desc.getCopy();
      data = src.data;
      valdef = src.valdef;
      f1 = src.f1;
      f2 = src.f2;
      f3 = src.f3;
      f4 = src.f4;
      f5 = src.f5;
      buffer = src.buffer;
      bytes = src.bytes;
    }

    public AbstractArray getCopy(IBJarr arr) throws Exception {
      if (arr == null || !structure.equals(arr.getStructure()))
        throw new Exception("incompatible structure");
      if (arr.isMapped()) throw new Exception("improper destination");
      ByteArray fa = arr.new ByteArray((Object) null);
      fa.init(desc);
      fa.valdef = valdef;
      boolean mapped = isMapped();
      int nd = structure.dims;
      int[] fi = structure.first;
      for (int i = 0; i < structure.length[0]; i++) {
        if (nd == 1) {
          fa.f1[i] = (mapped) ? get(fi[0] + i) : f1[i];
          continue;
        }
        for (int j = 0; j < structure.length[1]; j++) {
          if (nd == 2) {
            fa.f2[i][j] = (mapped) ? get(fi[0] + i, fi[1] + j) : f2[i][j];
            continue;
          }
          for (int k = 0; k < structure.length[2]; k++) {
            if (nd == 3) {
              fa.f3[i][j][k] = (mapped) ? get(fi[0] + i, fi[1] + j, fi[2] + k)
                  : f3[i][j][k];
              continue;
            }
            for (int l = 0; l < structure.length[3]; l++) {
              if (nd == 4) {
                fa.f4[i][j][k][l] = (mapped) ? get(fi[0] + i, fi[1] + j, fi[2]
                    + k, fi[3] + l) : f4[i][j][k][l];
                continue;
              }
              for (int m = 0; m < structure.length[4]; m++) {
                fa.f5[i][j][k][l][m] = (mapped) ? get(fi[0] + i, fi[1] + j,
                    fi[2] + k, fi[3] + l, fi[4] + m) : f5[i][j][k][l][m];
              }
            }
          }
        }
      }
      return fa;
    }

    /**
     * Set the value of an element.
     * @param f the value of the element.
     * @param ii index of the element. 
     * @throws Exception
     */
    public void set(byte f, int... ii) throws Exception {
      int[] i0 = structure.first;
      if (isMapped()) {
        ii = mapping.map(ii);
        i0 = mapping.dst.first;
      }
      else if (ii.length != structure.getDims())
        throw new Exception("invalid dimension");
      switch (ii.length) {
        case 1:
          f1[ii[0] - i0[0]] = f;
          break;
        case 2:
          f2[ii[0] - i0[0]][ii[1] - i0[1]] = f;
          break;
        case 3:
          f3[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]] = f;
          break;
        case 4:
          f4[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]] = f;
          break;
        case 5:
          f5[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]][ii[4]
              - i0[4]] = f;
          break;
      }
    }

    /**
     * Get the value of an element.
     * @param ii the index of the element.
     * @return the value.
     * @throws Exception
     */
    public byte get(int... ii) throws Exception {
      int[] i0 = structure.first;
      if (isMapped()) {
        ii = mapping.map(ii);
        i0 = mapping.dst.first;
      }
      else if (ii.length != structure.getDims())
        throw new Exception("invalid dimension");
      byte f = 0;
      switch (ii.length) {
        case 1:
          f = f1[ii[0] - i0[0]];
          break;
        case 2:
          f = f2[ii[0] - i0[0]][ii[1] - i0[1]];
          break;
        case 3:
          f = f3[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]];
          break;
        case 4:
          f = f4[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]];
          break;
        case 5:
          f = f5[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]][ii[4]
              - i0[4]];
          break;
      }
      return f;
    }

    public String format(int... ii) throws Exception {
      return format(get(ii));
    }

    /**
     * Get the formatted representation of a <i>BYTE</i> value. 
     * See {@link IBJarr.Descriptor Descriptor} for formatting options.
     * @param f the <i>BYTE</i> value.
     * @return  the formatted representation.
     * @throws Exception
     */
    public String format(byte f) throws Exception {
      return String.format(header.locale, desc.jfrm, f);
    }

    public void parse(String s, int... ii) throws Exception {
      byte i = Integer.valueOf(s, desc.radix).byteValue();
      set(i, ii);
    }

    public void fromBuffer(int... ii) throws Exception {
      set(bytes[0], ii);
    }

    public void toBuffer(int... ii) throws Exception {
      bytes[0] = get(ii);
    }
  }

  // ============================================================== ShortArray
  /**
   * A component array containing <i>SHORT</i> values.
   */
  public class ShortArray extends AbstractArray {
    short[] f1;
    short[][] f2;
    short[][][] f3;
    short[][][][] f4;
    short[][][][][] f5;

    private ShortArray(Object a) throws Exception {
      int[] ii = structure.getLength();
      switch (ii.length) {
        case 1:
          f1 = (a == null) ? new short[ii[0]] : (short[]) a;
          data = f1;
          break;
        case 2:
          f2 = (a == null) ? new short[ii[0]][ii[1]] : (short[][]) a;
          data = f2;
          break;
        case 3:
          f3 = (a == null) ? new short[ii[0]][ii[1]][ii[2]] : (short[][][]) a;
          data = f3;
          break;
        case 4:
          f4 = (a == null) ? new short[ii[0]][ii[1]][ii[2]][ii[3]]
              : (short[][][][]) a;
          data = f4;
          break;
        case 5:
          f5 = (a == null) ? new short[ii[0]][ii[1]][ii[2]][ii[3]][ii[4]]
              : (short[][][][][]) a;
          data = f5;
          break;
        default:
          throw new Exception("invalid dimension");
      }
    }

    private ShortArray(ShortArray src) {
      desc = src.desc.getCopy();
      data = src.data;
      valdef = src.valdef;
      f1 = src.f1;
      f2 = src.f2;
      f3 = src.f3;
      f4 = src.f4;
      f5 = src.f5;
      buffer = src.buffer;
      bytes = src.bytes;
    }

    /**
     * Provides a deep copy of this component array. The copy is not added
     * to the referenced array container.
     * @param arr the array container which the copy refers to.
     * @return deep copy of this component array.
     * @throws Exception
     */
    public AbstractArray getCopy(IBJarr arr) throws Exception {
      if (arr == null || !structure.equals(arr.getStructure()))
        throw new Exception("incompatible structure");
      if (arr.isMapped()) throw new Exception("improper destination");
      ShortArray fa = arr.new ShortArray((Object) null);
      fa.init(desc);
      fa.valdef = valdef;
      boolean mapped = isMapped();
      int nd = structure.dims;
      int[] fi = structure.first;
      for (int i = 0; i < structure.length[0]; i++) {
        if (nd == 1) {
          fa.f1[i] = (mapped) ? get(fi[0] + i) : f1[i];
          continue;
        }
        for (int j = 0; j < structure.length[1]; j++) {
          if (nd == 2) {
            fa.f2[i][j] = (mapped) ? get(fi[0] + i, fi[1] + j) : f2[i][j];
            continue;
          }
          for (int k = 0; k < structure.length[2]; k++) {
            if (nd == 3) {
              fa.f3[i][j][k] = (mapped) ? get(fi[0] + i, fi[1] + j, fi[2] + k)
                  : f3[i][j][k];
              continue;
            }
            for (int l = 0; l < structure.length[3]; l++) {
              if (nd == 4) {
                fa.f4[i][j][k][l] = (mapped) ? get(fi[0] + i, fi[1] + j, fi[2]
                    + k, fi[3] + l) : f4[i][j][k][l];
                continue;
              }
              for (int m = 0; m < structure.length[4]; m++) {
                fa.f5[i][j][k][l][m] = (mapped) ? get(fi[0] + i, fi[1] + j,
                    fi[2] + k, fi[3] + l, fi[4] + m) : f5[i][j][k][l][m];
              }
            }
          }
        }
      }
      return fa;
    }

    /**
     * Set the value of an element.
     * @param f the value of the element.
     * @param ii index of the element. 
     * @throws Exception
     */
    public void set(short f, int... ii) throws Exception {
      int[] i0 = structure.first;
      if (isMapped()) {
        ii = mapping.map(ii);
        i0 = mapping.dst.first;
      }
      else if (ii.length != structure.getDims())
        throw new Exception("invalid dimension");
      switch (ii.length) {
        case 1:
          f1[ii[0] - i0[0]] = f;
          break;
        case 2:
          f2[ii[0] - i0[0]][ii[1] - i0[1]] = f;
          break;
        case 3:
          f3[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]] = f;
          break;
        case 4:
          f4[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]] = f;
          break;
        case 5:
          f5[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]][ii[4]
              - i0[4]] = f;
          break;
      }
    }

    /**
     * Get the value of an element.
     * @param ii the index of the element.
     * @return the value.
     * @throws Exception
     */
    public short get(int... ii) throws Exception {
      int[] i0 = structure.first;
      if (isMapped()) {
        ii = mapping.map(ii);
        i0 = mapping.dst.first;
      }
      else if (ii.length != structure.getDims())
        throw new Exception("invalid dimension");
      short f = 0;
      switch (ii.length) {
        case 1:
          f = f1[ii[0] - i0[0]];
          break;
        case 2:
          f = f2[ii[0] - i0[0]][ii[1] - i0[1]];
          break;
        case 3:
          f = f3[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]];
          break;
        case 4:
          f = f4[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]];
          break;
        case 5:
          f = f5[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]][ii[4]
              - i0[4]];
          break;
      }
      return f;
    }

    public String format(int... ii) throws Exception {
      return format(get(ii));
    }

    /**
     * Get the formatted representation of a <i>SHORT</i> value. 
     * See {@link IBJarr.Descriptor Descriptor} for formatting options.
     * @param f the <i>SHORT</i> value.
     * @return  the formatted representation.
     * @throws Exception
     */
    public String format(int f) throws Exception {
      return String.format(header.locale, desc.jfrm, f);
    }

    public void parse(String s, int... ii) throws Exception {
      short i = Short.valueOf(s, desc.radix);
      set(i, ii);
    }

    public void fromBuffer(int... ii) throws Exception {
      short i = buffer.getShort(0);
      set(i, ii);
    }

    public void toBuffer(int... ii) throws Exception {
      short i = get(ii);
      buffer.putShort(0, i);
    }
  }

  // ============================================================== IntegerArray
  /**
   * A component array containing <i>INTEGER</i> values.
   */
  public class IntegerArray extends AbstractArray {
    int[] f1;
    int[][] f2;
    int[][][] f3;
    int[][][][] f4;
    int[][][][][] f5;

    private IntegerArray(Object a) throws Exception {
      int[] ii = structure.getLength();
      switch (ii.length) {
        case 1:
          f1 = (a == null) ? new int[ii[0]] : (int[]) a;
          data = f1;
          break;
        case 2:
          f2 = (a == null) ? new int[ii[0]][ii[1]] : (int[][]) a;
          data = f2;
          break;
        case 3:
          f3 = (a == null) ? new int[ii[0]][ii[1]][ii[2]] : (int[][][]) a;
          data = f3;
          break;
        case 4:
          f4 = (a == null) ? new int[ii[0]][ii[1]][ii[2]][ii[3]]
              : (int[][][][]) a;
          data = f4;
          break;
        case 5:
          f5 = (a == null) ? new int[ii[0]][ii[1]][ii[2]][ii[3]][ii[4]]
              : (int[][][][][]) a;
          data = f5;
          break;
        default:
          throw new Exception("invalid dimension");
      }
    }

    private IntegerArray(IntegerArray src) {
      desc = src.desc.getCopy();
      data = src.data;
      valdef = src.valdef;
      f1 = src.f1;
      f2 = src.f2;
      f3 = src.f3;
      f4 = src.f4;
      f5 = src.f5;
      buffer = src.buffer;
      bytes = src.bytes;
    }

    /**
     * Provides a deep copy of this component array. The copy is not added
     * to the referenced array container.
     * @param arr the array container which the copy refers to.
     * @return deep copy of this component array.
     * @throws Exception
     */
    public AbstractArray getCopy(IBJarr arr) throws Exception {
      if (arr == null || !structure.equals(arr.getStructure()))
        throw new Exception("incompatible structure");
      if (arr.isMapped()) throw new Exception("improper destination");
      IntegerArray fa = arr.new IntegerArray((Object) null);
      fa.init(desc);
      fa.valdef = valdef;
      boolean mapped = isMapped();
      int nd = structure.dims;
      int[] fi = structure.first;
      for (int i = 0; i < structure.length[0]; i++) {
        if (nd == 1) {
          fa.f1[i] = (mapped) ? get(fi[0] + i) : f1[i];
          continue;
        }
        for (int j = 0; j < structure.length[1]; j++) {
          if (nd == 2) {
            fa.f2[i][j] = (mapped) ? get(fi[0] + i, fi[1] + j) : f2[i][j];
            continue;
          }
          for (int k = 0; k < structure.length[2]; k++) {
            if (nd == 3) {
              fa.f3[i][j][k] = (mapped) ? get(fi[0] + i, fi[1] + j, fi[2] + k)
                  : f3[i][j][k];
              continue;
            }
            for (int l = 0; l < structure.length[3]; l++) {
              if (nd == 4) {
                fa.f4[i][j][k][l] = (mapped) ? get(fi[0] + i, fi[1] + j, fi[2]
                    + k, fi[3] + l) : f4[i][j][k][l];
                continue;
              }
              for (int m = 0; m < structure.length[4]; m++) {
                fa.f5[i][j][k][l][m] = (mapped) ? get(fi[0] + i, fi[1] + j,
                    fi[2] + k, fi[3] + l, fi[4] + m) : f5[i][j][k][l][m];
              }
            }
          }
        }
      }
      return fa;
    }

    /**
     * Set the value of an element.
     * @param f the value of the element.
     * @param ii index of the element. 
     * @throws Exception
     */
    public void set(int f, int... ii) throws Exception {
      int[] i0 = structure.first;
      if (isMapped()) {
        ii = mapping.map(ii);
        i0 = mapping.dst.first;
      }
      else if (ii.length != structure.getDims())
        throw new Exception("invalid dimension");
      switch (ii.length) {
        case 1:
          f1[ii[0] - i0[0]] = f;
          break;
        case 2:
          f2[ii[0] - i0[0]][ii[1] - i0[1]] = f;
          break;
        case 3:
          f3[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]] = f;
          break;
        case 4:
          f4[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]] = f;
          break;
        case 5:
          f5[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]][ii[4]
              - i0[4]] = f;
          break;
      }
    }

    /**
     * Get the value of an element.
     * @param ii the index of the element.
     * @return the value.
     * @throws Exception
     */
    public int get(int... ii) throws Exception {
      int[] i0 = structure.first;
      if (isMapped()) {
        ii = mapping.map(ii);
        i0 = mapping.dst.first;
      }
      else if (ii.length != structure.getDims())
        throw new Exception("invalid dimension");
      int f = 0;
      switch (ii.length) {
        case 1:
          f = f1[ii[0] - i0[0]];
          break;
        case 2:
          f = f2[ii[0] - i0[0]][ii[1] - i0[1]];
          break;
        case 3:
          f = f3[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]];
          break;
        case 4:
          f = f4[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]];
          break;
        case 5:
          f = f5[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]][ii[4]
              - i0[4]];
          break;
      }
      return f;
    }

    public String format(int... ii) throws Exception {
      return format(get(ii));
    }

    /**
     * Get the formatted representation of a <i>INTEGER</i> value. 
     * See {@link IBJarr.Descriptor Descriptor} for formatting options.
     * @param f the <i>INTEGER</i> value.
     * @return  the formatted representation.
     * @throws Exception
     */
    public String format(int f) throws Exception {
      return String.format(header.locale, desc.jfrm, f);
    }

    public void parse(String s, int... ii) throws Exception {
      int i = Integer.valueOf(s, desc.radix);
      set(i, ii);
    }

    public void fromBuffer(int... ii) throws Exception {
      int i = buffer.getInt(0);
      set(i, ii);
    }

    public void toBuffer(int... ii) throws Exception {
      int i = get(ii);
      buffer.putInt(0, i);
    }
  }

  // ============================================================== LongArray
  /**
   * A component array containing <i>LONG</i> values.
   */
  public class LongArray extends AbstractArray {
    long[] f1;
    long[][] f2;
    long[][][] f3;
    long[][][][] f4;
    long[][][][][] f5;

    private LongArray(Object a) throws Exception {
      int[] ii = structure.getLength();
      switch (ii.length) {
        case 1:
          f1 = (a == null) ? new long[ii[0]] : (long[]) a;
          data = f1;
          break;
        case 2:
          f2 = (a == null) ? new long[ii[0]][ii[1]] : (long[][]) a;
          data = f2;
          break;
        case 3:
          f3 = (a == null) ? new long[ii[0]][ii[1]][ii[2]] : (long[][][]) a;
          data = f3;
          break;
        case 4:
          f4 = (a == null) ? new long[ii[0]][ii[1]][ii[2]][ii[3]]
              : (long[][][][]) a;
          data = f4;
          break;
        case 5:
          f5 = (a == null) ? new long[ii[0]][ii[1]][ii[2]][ii[3]][ii[4]]
              : (long[][][][][]) a;
          data = f5;
          break;
        default:
          throw new Exception("invalid dimension");
      }
    }

    private LongArray(LongArray src) {
      desc = src.desc.getCopy();
      data = src.data;
      valdef = src.valdef;
      f1 = src.f1;
      f2 = src.f2;
      f3 = src.f3;
      f4 = src.f4;
      f5 = src.f5;
      buffer = src.buffer;
      bytes = src.bytes;
    }

    /**
     * Provides a deep copy of this component array. The copy is not added
     * to the referenced array container.
     * @param arr the array container which the copy refers to.
     * @return deep copy of this component array.
     * @throws Exception
     */
    public AbstractArray getCopy(IBJarr arr) throws Exception {
      if (arr == null || !structure.equals(arr.getStructure()))
        throw new Exception("incompatible structure");
      if (arr.isMapped()) throw new Exception("improper destination");
      LongArray fa = arr.new LongArray((Object) null);
      fa.init(desc);
      fa.valdef = valdef;
      boolean mapped = isMapped();
      int nd = structure.dims;
      int[] fi = structure.first;
      for (int i = 0; i < structure.length[0]; i++) {
        if (nd == 1) {
          fa.f1[i] = (mapped) ? get(fi[0] + i) : f1[i];
          continue;
        }
        for (int j = 0; j < structure.length[1]; j++) {
          if (nd == 2) {
            fa.f2[i][j] = (mapped) ? get(fi[0] + i, fi[1] + j) : f2[i][j];
            continue;
          }
          for (int k = 0; k < structure.length[2]; k++) {
            if (nd == 3) {
              fa.f3[i][j][k] = (mapped) ? get(fi[0] + i, fi[1] + j, fi[2] + k)
                  : f3[i][j][k];
              continue;
            }
            for (int l = 0; l < structure.length[3]; l++) {
              if (nd == 4) {
                fa.f4[i][j][k][l] = (mapped) ? get(fi[0] + i, fi[1] + j, fi[2]
                    + k, fi[3] + l) : f4[i][j][k][l];
                continue;
              }
              for (int m = 0; m < structure.length[4]; m++) {
                fa.f5[i][j][k][l][m] = (mapped) ? get(fi[0] + i, fi[1] + j,
                    fi[2] + k, fi[3] + l, fi[4] + m) : f5[i][j][k][l][m];
              }
            }
          }
        }
      }
      return fa;
    }

    /**
     * Set the value of an element.
     * @param f the value of the element.
     * @param ii index of the element. 
     * @throws Exception
     */
    public void set(long f, int... ii) throws Exception {
      int[] i0 = structure.first;
      if (isMapped()) {
        ii = mapping.map(ii);
        i0 = mapping.dst.first;
      }
      else if (ii.length != structure.getDims())
        throw new Exception("invalid dimension");
      switch (ii.length) {
        case 1:
          f1[ii[0] - i0[0]] = f;
          break;
        case 2:
          f2[ii[0] - i0[0]][ii[1] - i0[1]] = f;
          break;
        case 3:
          f3[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]] = f;
          break;
        case 4:
          f4[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]] = f;
          break;
        case 5:
          f5[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]][ii[4]
              - i0[4]] = f;
          break;
      }
    }

    /**
     * Get the value of an element.
     * @param ii the index of the element.
     * @return the value.
     * @throws Exception
     */
    public long get(int... ii) throws Exception {
      int[] i0 = structure.first;
      if (isMapped()) {
        ii = mapping.map(ii);
        i0 = mapping.dst.first;
      }
      else if (ii.length != structure.getDims())
        throw new Exception("invalid dimension");
      long f = 0;
      switch (ii.length) {
        case 1:
          f = f1[ii[0] - i0[0]];
          break;
        case 2:
          f = f2[ii[0] - i0[0]][ii[1] - i0[1]];
          break;
        case 3:
          f = f3[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]];
          break;
        case 4:
          f = f4[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]];
          break;
        case 5:
          f = f5[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]][ii[4]
              - i0[4]];
          break;
      }
      return f;
    }

    public String format(int... ii) throws Exception {
      return format(get(ii));
    }

    /**
     * Get the formatted representation of a <i>LONG</i> value. 
     * See {@link IBJarr.Descriptor Descriptor} for formatting options.
     * @param f the <i>LONG</i> value.
     * @return  the formatted representation.
     * @throws Exception
     */
    public String format(long f) throws Exception {
      return String.format(header.locale, desc.jfrm, f);
    }

    public void parse(String s, int... ii) throws Exception {
      long l = Long.valueOf(s, desc.radix);
      set(l, ii);
    }

    public void fromBuffer(int... ii) throws Exception {
      long l = buffer.getLong(0);
      set(l, ii);
    }

    public void toBuffer(int... ii) throws Exception {
      long l = get(ii);
      buffer.putLong(0, l);
    }
  }

  // ============================================================== FloatArray
  /**
   * A component array containing <i>FLOAT</i> values.
   */
  public class FloatArray extends AbstractArray {
    float[] f1;
    float[][] f2;
    float[][][] f3;
    float[][][][] f4;
    float[][][][][] f5;

    private FloatArray(Object a) throws Exception {
      int[] ii = structure.getLength();
      switch (ii.length) {
        case 1:
          f1 = (a == null) ? new float[ii[0]] : (float[]) a;
          data = f1;
          break;
        case 2:
          f2 = (a == null) ? new float[ii[0]][ii[1]] : (float[][]) a;
          data = f2;
          break;
        case 3:
          f3 = (a == null) ? new float[ii[0]][ii[1]][ii[2]] : (float[][][]) a;
          data = f3;
          break;
        case 4:
          f4 = (a == null) ? new float[ii[0]][ii[1]][ii[2]][ii[3]]
              : (float[][][][]) a;
          data = f4;
          break;
        case 5:
          f5 = (a == null) ? new float[ii[0]][ii[1]][ii[2]][ii[3]][ii[4]]
              : (float[][][][][]) a;
          data = f5;
          break;
        default:
          throw new Exception("invalid dimension");
      }
    }

    private FloatArray(FloatArray src) {
      desc = src.desc.getCopy();
      data = src.data;
      valdef = src.valdef;
      f1 = src.f1;
      f2 = src.f2;
      f3 = src.f3;
      f4 = src.f4;
      f5 = src.f5;
      buffer = src.buffer;
      bytes = src.bytes;
    }

    /**
     * Provides a deep copy of this component array. The copy is not added
     * to the referenced array container.
     * @param arr the array container which the copy refers to.
     * @return deep copy of this component array.
     * @throws Exception
     */
    public AbstractArray getCopy(IBJarr arr) throws Exception {
      if (arr == null || !structure.equals(arr.getStructure()))
        throw new Exception("incompatible structure");
      if (arr.isMapped())
        throw new Exception("improper destination");
      FloatArray fa = arr.new FloatArray((Object) null);
      fa.init(desc);
      fa.valdef = valdef;
      boolean mapped = isMapped();
      int nd = structure.dims;
      int[] fi = structure.first;
      for (int i = 0; i < structure.length[0]; i++) {
        if (nd == 1) {
          fa.f1[i] = (mapped) ? get(fi[0] + i) : f1[i];
          continue;
        }
        for (int j = 0; j < structure.length[1]; j++) {
          if (nd == 2) {
            fa.f2[i][j] = (mapped) ? get(fi[0] + i, fi[1] + j) : f2[i][j];
            continue;
          }
          for (int k = 0; k < structure.length[2]; k++) {
            if (nd == 3) {
              fa.f3[i][j][k] = (mapped) ? get(fi[0] + i, fi[1] + j, fi[2] + k)
                  : f3[i][j][k];
              continue;
            }
            for (int l = 0; l < structure.length[3]; l++) {
              if (nd == 4) {
                fa.f4[i][j][k][l] = (mapped) ? get(fi[0] + i, fi[1] + j, fi[2]
                    + k, fi[3] + l) : f4[i][j][k][l];
                continue;
              }
              for (int m = 0; m < structure.length[4]; m++) {
                fa.f5[i][j][k][l][m] = (mapped) ? get(fi[0] + i, fi[1] + j,
                    fi[2] + k, fi[3] + l, fi[4] + m) : f5[i][j][k][l][m];
              }
            }
          }
        }
      }
      return fa;
    }

    /**
     * Set the value of an element.
     * @param f the value of the element.
     * @param ii index of the element. 
     * @throws Exception
     */
    public void set(float f, int... ii) throws Exception {
      int[] i0 = structure.first;
      if (isMapped()) {
        ii = mapping.map(ii);
        i0 = mapping.dst.first;
      }
      else if (ii.length != structure.getDims())
        throw new Exception("invalid dimension");
      switch (ii.length) {
        case 1:
          f1[ii[0] - i0[0]] = f;
          break;
        case 2:
          f2[ii[0] - i0[0]][ii[1] - i0[1]] = f;
          break;
        case 3:
          f3[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]] = f;
          break;
        case 4:
          f4[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]] = f;
          break;
        case 5:
          f5[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]][ii[4]
              - i0[4]] = f;
          break;
      }
    }

    /**
     * Get the value of an element.
     * @param ii the index of the element.
     * @return the value.
     * @throws Exception
     */
    public float get(int... ii) throws Exception {
      int[] i0 = structure.first;
      if (isMapped()) {
        ii = mapping.map(ii);
        i0 = mapping.dst.first;
      }
      else if (ii.length != structure.getDims())
        throw new Exception("invalid dimension");
      float f = 0;
      switch (ii.length) {
        case 1:
          f = f1[ii[0] - i0[0]];
          break;
        case 2:
          f = f2[ii[0] - i0[0]][ii[1] - i0[1]];
          break;
        case 3:
          f = f3[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]];
          break;
        case 4:
          f = f4[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]];
          break;
        case 5:
          f = f5[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]][ii[4]
              - i0[4]];
          break;
      }
      return f;
    }

    public String format(int... ii) throws Exception {
      return format(get(ii));
    }

    /**
     * Get the formatted representation of a <i>FLOAT</i> value. 
     * See {@link IBJarr.Descriptor Descriptor} for formatting options.
     * @param f the <i>FLOAT</i> value.
     * @return  the formatted representation.
     * @throws Exception
     */
    public String format(float f) throws Exception {
      String s = String.format(header.locale, desc.jfrm, fact * f);
      if (header.locale == Locale.GERMAN)
        s = s.replace('.', ',');
      return s;
    }

    public void parse(String s, int... ii) throws Exception {
      float f = Float.valueOf(s.replace(',', '.')) / fact;
      set(f, ii);
    }

    public void fromBuffer(int... ii) throws Exception {
      float f = buffer.getFloat(0);
      set(f, ii);
    }

    public void toBuffer(int... ii) throws Exception {
      float f = get(ii);
      buffer.putFloat(0, f);
    }
  }

  // ============================================================== DoubleArray
  /**
   * A component array containing <i>DOUBLE</i> values.
   */
  public class DoubleArray extends AbstractArray {
    double[] f1;
    double[][] f2;
    double[][][] f3;
    double[][][][] f4;
    double[][][][][] f5;

    private DoubleArray(Object a) throws Exception {
      int[] ii = structure.getLength();
      switch (ii.length) {
        case 1:
          f1 = (a == null) ? new double[ii[0]] : (double[]) a;
          data = f1;
          break;
        case 2:
          f2 = (a == null) ? new double[ii[0]][ii[1]] : (double[][]) a;
          data = f2;
          break;
        case 3:
          f3 = (a == null) ? new double[ii[0]][ii[1]][ii[2]] : (double[][][]) a;
          data = f3;
          break;
        case 4:
          f4 = (a == null) ? new double[ii[0]][ii[1]][ii[2]][ii[3]]
              : (double[][][][]) a;
          data = f4;
          break;
        case 5:
          f5 = (a == null) ? new double[ii[0]][ii[1]][ii[2]][ii[3]][ii[4]]
              : (double[][][][][]) a;
          data = f5;
          break;
        default:
          throw new Exception("invalid dimension");
      }
    }

    private DoubleArray(DoubleArray src) {
      desc = src.desc.getCopy();
      data = src.data;
      valdef = src.valdef;
      f1 = src.f1;
      f2 = src.f2;
      f3 = src.f3;
      f4 = src.f4;
      f5 = src.f5;
      buffer = src.buffer;
      bytes = src.bytes;
    }

    /**
     * Provides a deep copy of this component array. The copy is not added
     * to the referenced array container.
     * @param arr the array container which the copy refers to.
     * @return deep copy of this component array.
     * @throws Exception
     */
    public AbstractArray getCopy(IBJarr arr) throws Exception {
      if (arr == null || !structure.equals(arr.getStructure()))
        throw new Exception("incompatible structure");
      if (arr.isMapped()) throw new Exception("improper destination");
      DoubleArray fa = arr.new DoubleArray((Object) null);
      fa.init(desc);
      fa.valdef = valdef;
      boolean mapped = isMapped();
      int nd = structure.dims;
      int[] fi = structure.first;
      for (int i = 0; i < structure.length[0]; i++) {
        if (nd == 1) {
          fa.f1[i] = (mapped) ? get(fi[0] + i) : f1[i];
          continue;
        }
        for (int j = 0; j < structure.length[1]; j++) {
          if (nd == 2) {
            fa.f2[i][j] = (mapped) ? get(fi[0] + i, fi[1] + j) : f2[i][j];
            continue;
          }
          for (int k = 0; k < structure.length[2]; k++) {
            if (nd == 3) {
              fa.f3[i][j][k] = (mapped) ? get(fi[0] + i, fi[1] + j, fi[2] + k)
                  : f3[i][j][k];
              continue;
            }
            for (int l = 0; l < structure.length[3]; l++) {
              if (nd == 4) {
                fa.f4[i][j][k][l] = (mapped) ? get(fi[0] + i, fi[1] + j, fi[2]
                    + k, fi[3] + l) : f4[i][j][k][l];
                continue;
              }
              for (int m = 0; m < structure.length[4]; m++) {
                fa.f5[i][j][k][l][m] = (mapped) ? get(fi[0] + i, fi[1] + j,
                    fi[2] + k, fi[3] + l, fi[4] + m) : f5[i][j][k][l][m];
              }
            }
          }
        }
      }
      return fa;
    }

    /**
     * Set the value of an element.
     * @param f the value of the element.
     * @param ii index of the element. 
     * @throws Exception
     */
    public void set(double f, int... ii) throws Exception {
      int[] i0 = structure.first;
      if (isMapped()) {
        ii = mapping.map(ii);
        i0 = mapping.dst.first;
      }
      else if (ii.length != structure.getDims())
        throw new Exception("invalid dimension");
      switch (ii.length) {
        case 1:
          f1[ii[0] - i0[0]] = f;
          break;
        case 2:
          f2[ii[0] - i0[0]][ii[1] - i0[1]] = f;
          break;
        case 3:
          f3[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]] = f;
          break;
        case 4:
          f4[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]] = f;
          break;
        case 5:
          f5[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]][ii[4]
              - i0[4]] = f;
          break;
      }
    }

    /**
     * Get the value of an element.
     * @param ii the index of the element.
     * @return the value.
     * @throws Exception
     */
    public double get(int... ii) throws Exception {
      int[] i0 = structure.first;
      if (isMapped()) {
        ii = mapping.map(ii);
        i0 = mapping.dst.first;
      }
      else if (ii.length != structure.getDims())
        throw new Exception("invalid dimension");
      double f = 0;
      switch (ii.length) {
        case 1:
          f = f1[ii[0] - i0[0]];
          break;
        case 2:
          f = f2[ii[0] - i0[0]][ii[1] - i0[1]];
          break;
        case 3:
          f = f3[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]];
          break;
        case 4:
          f = f4[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]];
          break;
        case 5:
          f = f5[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]][ii[4]
              - i0[4]];
          break;
      }
      return f;
    }

    public String format(int... ii) throws Exception {
      return format(get(ii));
    }

    /**
     * Get the formatted representation of a <i>DOUBLE</i> value. 
     * See {@link IBJarr.Descriptor Descriptor} for formatting options.
     * @param f the <i>DOUBLE</i> value.
     * @return  the formatted representation.
     * @throws Exception
     */
    public String format(double f) throws Exception {
      String s = String.format(header.locale, desc.jfrm, fact * f); 
      if (header.locale == Locale.GERMAN)
        s = s.replace('.', ',');
      return s;
    }

    public void parse(String s, int... ii) throws Exception {
      double d = Double.valueOf(s.replace(',', '.')) / fact;
      set(d, ii);
    }

    public void fromBuffer(int... ii) throws Exception {
      double d = buffer.getDouble(0);
      set(d, ii);
    }

    public void toBuffer(int... ii) throws Exception {
      double d = get(ii);
      buffer.putDouble(0, d);
    }
  }

  // ============================================================== StringArray
  /**
   * A component array containing <i>STRING</i> values.
   */
  public class StringArray extends AbstractArray {
    String[] f1;
    String[][] f2;
    String[][][] f3;
    String[][][][] f4;
    String[][][][][] f5;

    private StringArray(Object a) throws Exception {
      int[] ii = structure.getLength();
      switch (ii.length) {
        case 1:
          f1 = (a == null) ? new String[ii[0]] : (String[]) a;
          data = f1;
          break;
        case 2:
          f2 = (a == null) ? new String[ii[0]][ii[1]] : (String[][]) a;
          data = f2;
          break;
        case 3:
          f3 = (a == null) ? new String[ii[0]][ii[1]][ii[2]] : (String[][][]) a;
          data = f3;
          break;
        case 4:
          f4 = (a == null) ? new String[ii[0]][ii[1]][ii[2]][ii[3]]
              : (String[][][][]) a;
          data = f4;
          break;
        case 5:
          f5 = (a == null) ? new String[ii[0]][ii[1]][ii[2]][ii[3]][ii[4]]
              : (String[][][][][]) a;
          data = f5;
          break;
        default:
          throw new Exception("invalid dimension");
      }
    }

    private StringArray(StringArray src) {
      desc = src.desc.getCopy();
      data = src.data;
      valdef = src.valdef;
      f1 = src.f1;
      f2 = src.f2;
      f3 = src.f3;
      f4 = src.f4;
      f5 = src.f5;
      buffer = src.buffer;
      bytes = src.bytes;
    }

    /**
     * Provides a deep copy of this component array. The copy is not added
     * to the referenced array container.
     * @param arr the array container which the copy refers to.
     * @return deep copy of this component array.
     * @throws Exception
     */
    public AbstractArray getCopy(IBJarr arr) throws Exception {
      if (arr == null || !structure.equals(arr.getStructure()))
        throw new Exception("incompatible structure");
      if (arr.isMapped()) throw new Exception("improper destination");
      StringArray fa = arr.new StringArray((Object) null);
      fa.init(desc);
      fa.valdef = valdef;
      boolean mapped = isMapped();
      int nd = structure.dims;
      int[] fi = structure.first;
      for (int i = 0; i < structure.length[0]; i++) {
        if (nd == 1) {
          fa.f1[i] = (mapped) ? get(fi[0] + i) : f1[i];
          continue;
        }
        for (int j = 0; j < structure.length[1]; j++) {
          if (nd == 2) {
            fa.f2[i][j] = (mapped) ? get(fi[0] + i, fi[1] + j) : f2[i][j];
            continue;
          }
          for (int k = 0; k < structure.length[2]; k++) {
            if (nd == 3) {
              fa.f3[i][j][k] = (mapped) ? get(fi[0] + i, fi[1] + j, fi[2] + k)
                  : f3[i][j][k];
              continue;
            }
            for (int l = 0; l < structure.length[3]; l++) {
              if (nd == 4) {
                fa.f4[i][j][k][l] = (mapped) ? get(fi[0] + i, fi[1] + j, fi[2]
                    + k, fi[3] + l) : f4[i][j][k][l];
                continue;
              }
              for (int m = 0; m < structure.length[4]; m++) {
                fa.f5[i][j][k][l][m] = (mapped) ? get(fi[0] + i, fi[1] + j,
                    fi[2] + k, fi[3] + l, fi[4] + m) : f5[i][j][k][l][m];
              }
            }
          }
        }
      }
      return fa;
    }

    /**
     * Set the value of an element.
     * @param f the value of the element.
     * @param ii index of the element. 
     * @throws Exception
     */
    public void set(String f, int... ii) throws Exception {
      int[] i0 = structure.first;
      if (isMapped()) {
        ii = mapping.map(ii);
        i0 = mapping.dst.first;
      }
      else if (ii.length != structure.getDims())
        throw new Exception("invalid dimension");
      switch (ii.length) {
        case 1:
          f1[ii[0] - i0[0]] = f;
          break;
        case 2:
          f2[ii[0] - i0[0]][ii[1] - i0[1]] = f;
          break;
        case 3:
          f3[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]] = f;
          break;
        case 4:
          f4[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]] = f;
          break;
        case 5:
          f5[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]][ii[4]
              - i0[4]] = f;
          break;
      }
    }

    /**
     * Get the value of an element.
     * @param ii the index of the element.
     * @return the value.
     * @throws Exception
     */
    public String get(int... ii) throws Exception {
      int[] i0 = structure.first;
      if (isMapped()) {
        ii = mapping.map(ii);
        i0 = mapping.dst.first;
      }
      else if (ii.length != structure.getDims())
        throw new Exception("invalid dimension");
      String f = null;
      switch (ii.length) {
        case 1:
          f = f1[ii[0] - i0[0]];
          break;
        case 2:
          f = f2[ii[0] - i0[0]][ii[1] - i0[1]];
          break;
        case 3:
          f = f3[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]];
          break;
        case 4:
          f = f4[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]];
          break;
        case 5:
          f = f5[ii[0] - i0[0]][ii[1] - i0[1]][ii[2] - i0[2]][ii[3] - i0[3]][ii[4]
              - i0[4]];
          break;
      }
      return f;
    }

    public String format(int... ii) throws Exception {
      return format(get(ii));
    }

    /**
     * Get the formatted representation of a <i>STRING</i> value. 
     * See {@link IBJarr.Descriptor Descriptor} for formatting options.
     * @param f the <i>STRING</i> value.
     * @return  the formatted representation.
     * @throws Exception
     */
    public String format(String f) throws Exception {
      return String.format(header.locale, desc.jfrm, f);
    }

    public void parse(String s, int... ii) throws Exception {
      s = IBJhdr.unquote(s);
      if (s.length() > desc.size) s = s.substring(0, desc.size);
      set(s, ii);
    }

    public void fromBuffer(int... ii) throws Exception {
      String s = new String(bytes, charset.name());
      s = s.trim();
      set(s, ii);
    }

    public void toBuffer(int... ii) throws Exception {
      String s = get(ii);
      byte[] bb = s.getBytes(charset.name());
      if (bb.length > bytes.length) throw new Exception("buffer overflow");
      for (int i = 0; i < bb.length; i++)
        bytes[i] = bb[i];
      for (int i = bb.length; i < bytes.length; i++)
        bytes[i] = 0;
    }
  }

  // ============================================================== TimeArray
  /**
   * A component array containing <i>TIME</i> values (stored as <i>INTEGER</i>).
   */
  public class TimeArray extends IntegerArray {
    private TimeArray(Object a) throws Exception {
      super(a);
    }

    private TimeArray(TimeArray src) {
      super(src);
    }

    /**
     * Get the formatted representation of a <i>TIME</i> value using the 
     * format DDD.HH:mm:ss. 
     * @param f the <i>TIME</i> value.
     * @return  the formatted representation.
     * @throws Exception
     */
    @Override
    public String format(int f) throws Exception {
      return IBJhdr.formatTime(f);
    }

    @Override
    public String format(int... ii) throws Exception {
      return IBJhdr.formatTime(get(ii));
    }

    @Override
    public void parse(String s, int... ii) throws Exception {
      int i = IBJhdr.parseTime(s);
      set(i, ii);
    }
  }

  // ============================================================== DateArray
  /**
   * A component array containing <i>DATE</i> values (stored as <i>DOUBLE</i>).
   */
  public class DateArray extends DoubleArray {
    private DateArray(Object a) throws Exception {
      super(a);
    }

    private DateArray(DateArray src) {
      super(src);
    }

    @Override
    public String format(int... ii) throws Exception {
      return format(get(ii));
    }

    /**
     * Get the formatted representation of a <i>DATE</i> value using the
     * format yyyy-MM-dd.HH:mm:ss. 
     * @param d the <i>DATE</i> value.
     * @return  the formatted representation.
     * @throws Exception
     */
    @Override
    public String format(double d) throws Exception {
      return IBJhdr.formatDate(d, header.timezone);
    }

    @Override
    public void parse(String s, int... ii) throws Exception {
      double d = IBJhdr.parseDate(s, header.timezone);
      set(d, ii);
    }
  }

  // ==========================================================================
  /**
   * <p>
   * Data descriptor with information on the data type and how to generate a 
   * formatted representation of the data. Valid data types are enumerated in
   * {@link IBJdcl IBJdcl}.
   * A {@code Descriptor} is derived from a format specification similar to
   * that used in "C". The format consists of:
   * </p>
   * <p>
   * <i>Name</i>{@code %}[<i>Flag</i>]<i>Width</i>[.<i>Precision</i>][<i>Modifier</i>]<i>Type</i>
   * </p>
   * <p>
   * The meaning of the fields is:<br>
   * <table cellpadding="4">
   * <tr><th>Name</th><th>Description</th></tr>
   * <tr><td valign="top"><i>Name</i></td>
   * <td>The name by which the component array containing these data may be 
   * retrieved. Whithin an {@code IBJarr} the names must be unique.</td>
   * </tr>
   * <tr><td valign="top"><i>Flag</i></td>
   * <td>An optional minus sign to indicate that the generated string should
   * be left adjusted. In floating point formats this field optionally 
   * starts with a scaling factor used for formatted output. The format
   * of the scaling factor is {@code (*}<i>Factor</i>{@code )}.</td>
   * </tr>
   * <tr><td valign="top"><i>Width</i></td>
   * <td>The number of characters to be used for the formatted output. If the 
   * representation of the data requires mode characters the output width is 
   * increased.</td>
   * </tr>
   * <tr><td valign="top"><i>Precision</i></td>
   * <td>For floating point data: the number of digits to the right of the
   * decimal point. For string data: the maximum field width and the
   * number of bytes to be used for a binary representation.</td>
   * </tr>
   * <tr><td valign="top"><i>Modifier</i></td>
   * <td>Additional (optional) information on the data type (see next table).</td>
   * </tr>
   * <tr><td valign="top"><i>Type</i></td>
   * <td>Specification of the data type (see next table).</td>
   * </tr>
   * </table>
   * </p>
   * <p>
   * The next table shows how <i>Modifier</i> and <i>Type</i> determine the
   * data type:<br/>
   * <table cellpadding="4" align="center">
   * <tr><th><i>Type</i></th>
   * <th align="center"><i>Modifier</i></th>
   * <th>{@code IBJdcl.DataType}</th>
   * <th>JAVA type</th>
   * <th>Size</th>
   * </tr>
   * <tr><td>{@code d | x}</td><td align="center">{@code b}</td>
   * <td><code><i>BYTE</i></code></td><td>{@code byte}</td><td align="center">1</td>
   * </tr>
   * <tr><td>{@code d | x}</td><td align="center">{@code h}</td>
   * <td><code><i>SHORT</i></code></td><td>{@code short}</td><td align="center">2</td>
   * </tr>
   * <tr><td>{@code d | x}</td><td align="center"></td>
   * <td><code><i>INTEGER</i></code></td><td>{@code int}</td><td align="center">4</td>
   * </tr>
   * <tr><td>{@code d | x}</td><td align="center">{@code L}</td>
   * <td><code><i>LONG</i></code></td><td>{@code long}</td><td align="center">8</td>
   * </tr>
   * <tr><td>{@code f | e}</td><td align="center"></td>
   * <td><code><i>FLOAT</i></code></td><td>{@code float}</td><td align="center">4</td>
   * </tr>
   * <tr><td>{@code f | e}</td><td align="center">{@code l}</td>
   * <td><code><i>DOUBLE</i></code></td><td>{@code double}</td><td align="center">8</td>
   * </tr>
   * <tr><td>{@code t}</td><td align="center"></td>
   * <td><code><i>TIME</i></code></td><td>{@code int}</td><td align="center">4</td>
   * </tr>
   * <tr><td>{@code t}</td><td align="center">{@code l}</td>
   * <td><code><i>DATE</i></code></td><td>{@code double}</td><td align="center">8</td>
   * </tr>
   * <tr><td>{@code s}</td><td align="center"></td>
   * <td><code><i>STRING</i></code></td><td>{@code String}</td><td><i>Precision</i></td>
   * </tr>
   * </table>
   * </p>
   */
  public static class Descriptor {
    String format; // the original format
    String name; // the name associated with this array
    String cfrm; // C-format used in the header
    float fact; // scaling factor (for float or double)
    String flag; // format flag
    String width; // format width
    String prcsn; // format precision
    String modfr; // format modifier
    char cnvc; // conversion character
    int size; // number of bytes in the binary file or number of characters
    int length; // number of bytes in the binary file
    int radix; // radix for reading integer values
    DataType type; // the data type corresponding to the given format
    Class clss; // the java class of the elements
    String jfrm; // format for printing

    private Descriptor() {
      fact = Float.NaN;
      width = "";
      prcsn = "";
      modfr = "";
      flag = "";
      radix = 10;
    }

    /**
     * Create a data descriptor from the format specification.
     * @param frm the format specification (see {@link IBJarr.Descriptor Descriptor}).
     * @throws Exception
     */
    public Descriptor(String frm) throws Exception {
      this();
      format = frm;
      try {
        int i = 0;
        int l = frm.length();
        int ip = frm.indexOf('%');
        name = frm.substring(0, ip);
        cfrm = frm.substring(ip);
        i = ip + 1;
        if (frm.charAt(i) == '(') {
          for (; i < l; i++)
            if (frm.charAt(i) == ')')
              break;
          if (frm.charAt(ip + 2) == '*')
            fact = Float.valueOf(frm.substring(ip + 3, i));
          i++;
        }
        ip = i;
        if ("+- ".indexOf(frm.charAt(i)) >= 0) {
          flag = frm.substring(i, i + 1);
          ip++;
        }
        for (i = ip; i < l; i++)
          if (!Character.isDigit(frm.charAt(i)))
            break;
        if (i > ip) {
          width = frm.substring(ip, i);
          if (frm.charAt(i) == '.') {
            ip = i;
            for (i = ip + 1; i < l; i++)
              if (!Character.isDigit(frm.charAt(i)))
                break;
            prcsn = frm.substring(ip, i);
          }
        }
        ip = i;
        i = l - 1;
        if (ip <= i) modfr = frm.substring(ip, i);
        cnvc = frm.charAt(i);
        //
        jfrm = String.format("%%%s%s%s%c", flag, width, prcsn, cnvc);
        switch (cnvc) {
          case 'x':
            radix = 16;
          case 'd':
            if (modfr.equals("b")) {
              size = 1;
              type = DataType.BYTE;
              clss = byte.class;
            }
            else if (modfr.equals("h")) {
              size = 2;
              type = DataType.SHORT;
              clss = short.class;
            }
            else if (modfr.equals("L")) {
              size = 8;
              type = DataType.LONG;
              clss = long.class;
            }
            else {
              size = 4;
              type = DataType.INTEGER;
              clss = int.class;
            }
            break;
          case 'f':
          case 'e':
            if (modfr.equals("l")) {
              size = 8;
              type = DataType.DOUBLE;
              clss = double.class;
            }
            else {
              size = 4;
              type = DataType.FLOAT;
              clss = float.class;
            }
            break;
          case 's':
            if (prcsn.length() > 1) size = Integer.valueOf(prcsn.substring(1));
            else size = Integer.valueOf(width);
            type = DataType.STRING;
            clss = String.class;
            break;
          case 't':
            if (modfr.equals("l")) {
              size = 8;
              type = DataType.DATE;
              clss = long.class;
              jfrm = "%tF.%<tT";
            }
            else {
              size = 4;
              type = DataType.TIME;
              clss = int.class;
              jfrm = "%8d";
            }
            break;
          default:
            throw new Exception("invalid conversion");
        }
        length = size;
      }
      catch (Exception e) {
        if (CHECK) e.printStackTrace();
        throw new Exception("invalid format \"" + frm + "\"");
      }
    }

    private Descriptor getCopy() {
      Descriptor desc = new Descriptor();
      desc.cfrm = cfrm;
      desc.clss = clss;
      desc.cnvc = cnvc;
      desc.fact = fact;
      desc.flag = flag;
      desc.jfrm = jfrm;
      desc.modfr = modfr;
      desc.name = name;
      desc.prcsn = prcsn;
      desc.size = size;
      desc.length = length;
      desc.type = type;
      desc.width = width;
      desc.format = format;
      desc.radix = radix;
      return desc;
    }

    /**
     * Splits up a string containing C-formats. Square brackets are translated
     * into subsequent format strings.
     * 
     * @param line the string to split
     * @return the array of format strings
     */
    public static String[] expandFormat(String line) throws Exception {
      String[] sa;
      String s, ss;
      StringBuffer sb;
      String form = "tefdscx";
      Vector<String> v = new Vector<String>();
      if (line == null) throw new Exception("no format string");
      int i, j, k, n, len;
      sb = new StringBuffer();
      for (i = 0; i < line.length(); i++)
        if (line.charAt(i) != '\"' && line.charAt(i) != ' ')
          sb.append(line.charAt(i));
      line = sb.toString();
      while ((len = line.length()) > 0) {
        i = line.indexOf('%');
        if (i < 0 || i >= len - 1)
          throw new Exception("missing format specifier");
        for (j = i + 1, k = 0; j < len; j++) {
          if (line.charAt(j) == '(') k++;
          if (line.charAt(j) == ')') k--;
          if (form.indexOf(line.charAt(j)) > -1 && k <= 0) break;
        }
        if (j == len) throw new Exception("missing conversion specifier");
        s = line.substring(0, j + 1).trim();
        if ((i = s.indexOf('[')) > -1) {
          if ((k = s.indexOf(']')) < 0)
            throw new Exception("missing closing bracket ']'");
          try {
            n = Integer.parseInt(s.substring(i + 1, k));
          }
          catch (Exception e) {
            throw new Exception("can't parse count");
          }
          ss = s.substring(0, i) + s.substring(k + 1);
          i = ss.indexOf('%') - 1;
          for (k = 0; k < n; k++) {
            sb = new StringBuffer(ss);
            sb.setCharAt(i, (char) ((int) ss.charAt(i) + k));
            v.addElement(sb.toString());
          }
        }
        else v.addElement(s);
        line = line.substring(j + 1);
      }
      sa = v.toArray(new String[] {});
      return sa;
    }
  }

  //===========================================================================

  public void printMapping(String title, PrintWriter pw) {
    if (mapping == null) {
      pw.printf("%s : not mapped\n", title);
      return;
    }
    mapping.print(title, pw);
  }

  // ======================================================================
  private static void listMapping(Mapping mm, PrintWriter pw) {
    try {
      pw.println("\nmapping:");
      int ns = mm.src.dims;
      int[] ii = (int[]) mm.src.first.clone();
      int i = 0;
      while (i >= 0) {
        int[] jj = mm.map(ii);
        for (int k = 0; k < ii.length; k++)
          pw.printf(" %2d", ii[k]);
        pw.printf(" --> ");
        for (int k = 0; k < jj.length; k++)
          pw.printf(" %2d", jj[k]);
        pw.println();
        for (i = ns - 1; i >= 0; i--) {
          ii[i]++;
          if (ii[i] < mm.src.first[i] + mm.src.length[i]) break;
          else ii[i] = mm.src.first[i];
        }
      }
      pw.flush();
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }
  
  private static void test01() {
    try {
      IBJarr arr = new IBJarr("test", 3, 4);
      arr.setFirstIndex(1, 1);
      arr.setTimeZone(null);
      IBJhdr hdr = arr.getHeader();
      hdr.putString("title", "A simple example", true);
      hdr.putDate("created", IBJhdr.getDate(new Date()));
      arr.createArray("z0%(*100)5.2f");
      FloatArray fa_z0 = (FloatArray)arr.getArray("z0");
      arr.createArray("d0%5.2f");
      FloatArray fa_d0 = (FloatArray)arr.getArray("d0");
      //
      for (int i=1; i<=3; i++) {
        for (int j=1; j<=4; j++) {
          fa_z0.set(0.1f*i, i, j);
        }
      }
      //
      float[][] z0 = (float[][])fa_z0.getData();
      float[][] d0 = (float[][])fa_d0.getData();
      for (int i=0; i<3; i++)
        for (int j=0; j<4; j++)
          d0[i][j] = 6*z0[i][j];
      //
      PrintWriter pw = new PrintWriter(new OutputStreamWriter(System.out));
      arr.printInfo(pw);
      pw.println("\nHeader:");
      hdr.print(pw);
      arr.printAll("\nThe data of "+hdr.getString("title", false), pw);
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    System.exit(0);
  }

  /**
   * For testing only.
   * @param args (not used)
   */
  public static void main(String[] args) {
    IBJarr arr;
    IBJarr.FloatArray fa, ga;
    float[][] ff;
    
    test01();
    
    try {
      // Object o = Array.newInstance(float.class, new int[] { 3, 5 });
      // float[][] a = (float[][]) o;
      // System.out.println("length=" + a.length);
      PrintWriter out = new PrintWriter(System.out);
      arr = new IBJarr("ORG", 4, 3);
      //
      boolean check = arr.checkStructure(new int[4][3]);
      out.println("check=" + check);
      //
      arr.getStructure().setFirstIndex(1, 1);
      fa = (IBJarr.FloatArray) arr.createArray("ff%6.1f");
      for (int i = 1; i <= 4; i++) {
        for (int j = 1; j <= 3; j++) {
          fa.set(10 * i + j, i, j);
        }
      }
      ff = (float[][]) fa.getData();
      for (int i = 0; i < 4; i++) {
        for (int j = 0; j < 3; j++)
          out.printf(fa.desc.jfrm, ff[i][j]);
        out.println();
      }
      fa.print("float array ff", out);
      //
      float[][] gg = new float[][] { { 11, 12, 13 }, { 21, 22, 23 },
          { 31, 32, 33 }, { 41, 42, 43 } };
      ga = (IBJarr.FloatArray) arr.createArray("gg%6.0f", gg);
      ga.print("float array gg", out);
      //
      arr.printAll("all arrays", out);
      //
      String s = "j-;i=2;";
      Mapping mm = new Mapping(arr.structure, s, null);
      mm.print("mapping " + s, out);
      listMapping(mm, out);
      IBJarr mrr = arr.getSelected("MAPPED", s);
      mrr.printInfo(out);
      FloatArray fm = (FloatArray) mrr.getArray("ff");
      fm.print("mapped:", out);
      IBJarr crr = mrr.getCopy("COPY");
      crr.printInfo(out);
      FloatArray fc = (FloatArray) crr.getArray("ff");
      fc.print("copied:", out);
      //
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }
}
