// =============================================================== eval-emsl.c
//
// Evaluation of validation emsl (cooling tower, KKW Emsland)
// ==========================================================
//
// Copyright (C) Umweltbundesamt, 14191 Berlin, Germany, 2004
// Copyright (C) Janicke Consulting, 26427 Dunum, Germany, 2004
// email: info@austal2000.de
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of
// the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// last change: 2004-09-08 uj
//
//============================================================================

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#define CC(i,j)  Cc[(i-1)*Ny+j-1]
#define DD(i,j)  Dd[(i-1)*Ny+j-1]

static char *FileCnc = "xx-j00z.dmna";
static char *FileDev = "xx-j00s.dmna";
static double *Cc, *Dd;
static double Xq=0, Yq=0;
static int Nx, Ny;

int equals(double f1, double f2) {
  return (fabs(f1-f2) < 0.001) ? 1 : 0;
}

int getMaximum(char *path, double *x, double *y, double *cm, double *dev) {
  char s[256], fn[256], buf[4000]="", *pc;
  int i, j, ii, jj, nx, ny;
  double c, d, x0, y0, dd;
  FILE *f;
  sprintf(fn, "%s/%s", path, FileCnc);
  f = fopen(fn, "r");                           // Datei ffnen
  if (!f) {
    printf("*** Datei \"%s\" kann nicht gelesen werden!\n", fn);
    exit(1);
  }
  while(fgets(buf, 4000, f)) {                  // Header berspringen
    for (i=0; i<strlen(buf); i++) buf[i] = tolower(buf[i]);
    if (!strncmp(buf,"xmin", 4)) {
      sscanf(buf+4, "%lf", &x0);  
    }
    if (!strncmp(buf,"ymin", 4)) {
      sscanf(buf+4, "%lf", &y0);    
    }
    if (!strncmp(buf,"delta", 5)) {
      sscanf(buf+5, "%lf", &dd);    
    }
    if (!strncmp(buf,"hghb", 4)) {
      sscanf(buf+4, "%d%d", &Nx, &Ny);
      Cc = malloc(Nx*Ny*sizeof(double));
    }
    if (*buf == '*')  break;
  }
  if (*buf != '*') {
    printf("*** Datenteil in Datei \"%s\" nicht gefunden!\n", fn);
    exit(2);
  }
  for (j=Ny; j>0; ) {
    if (!fgets(buf, 4000, f)) {                 // Zeilenweise einlesen
      printf("*** EOF fr j=%d gefunden!\n", j);
      exit(3);
    }
    pc = strtok(buf, " \t\r\n");
    if (!pc)  continue;
    for (i=1; i<=Nx; i++) {
      if (!pc) {
        printf("Datenelement (%d,%d) nicht gefunden!\n", i, j);
        exit(5);
      }
      if (1 != sscanf(pc, "%lf", &c))  {
        printf("Datenelement (%d,%d) nicht lesbar!\n", i, j);
        exit(4);
      }
      CC(i,j) = c;
      pc = strtok(NULL, " \t\r\n");
    } // for i
    j--;
  } // for j
  fclose(f);
  f = NULL;

  //

  sprintf(fn, "%s/%s", path, FileDev);
  f = fopen(fn, "r");                           // Datei ffnen
  if (!f) {
    printf("*** Datei \"%s\" kann nicht gelesen werden!\n", fn);
    exit(1);
  }
  while(fgets(buf, 4000, f)) {                  // Header berspringen
    if (!strncmp(buf,"hghb", 4)) {
      sscanf(buf+4, "%d%d", &nx, &ny);
      if (Nx!=nx || Ny!=ny) {
        printf("*** Ungltige Dateistruktur in \"%s\"!", fn);
        exit(10);
      }
      Dd = malloc(Nx*Ny*sizeof(double));
    }
    if (*buf == '*')  break;
  }
  if (*buf != '*') {
    printf("*** Datenteil in Datei \"%s\" nicht gefunden!\n", fn);
    exit(2);
  }
  for (j=Ny; j>0; ) {
    if (!fgets(buf, 4000, f)) {                 // Zeilenweise einlesen
      printf("*** EOF fr j=%d gefunden!\n", j);
      exit(3);
    }
    pc = strtok(buf, " \t\r\n");
    if (!pc)  continue;
    for (i=1; i<=Nx; i++) {
      if (!pc) {
        printf("Datenelement (%d,%d) nicht gefunden!\n", i, j);
        exit(5);
      }
      if (1 != sscanf(pc, "%lf", &d))  {
        printf("Datenelement (%d,%d) nicht lesbar!\n", i, j);
        exit(4);
      }
      DD(i,j) = d/100.;
      pc = strtok(NULL, " \t\r\n");
    } // for i
    j--;
  } // for j
  fclose(f);
  f = NULL;
  c = 0;
  for (i=1; i<=Nx; i++) 
    for (j=1; j<=Ny; j++) 
      if (CC(i,j) > c) { c = CC(i,j); ii = i; jj = j; }
  *cm = CC(ii,jj);
  *dev = DD(ii,jj);
  *x = x0 + (ii-0.5)*dd;
  *y = y0 + (jj-0.5)*dd;
  free(Cc);
  free(Dd); 
  return 0;
}

//==================================================================== main

int main( int argc, char *argv[] ) {
  double x, y, c0, d0, c, d;
  char path[256];
  
  if (argc < 2) {
    printf("Aufruf: eval-emsl <Ordner>\n");
    exit(0);
  }
  
  printf("Test %s:\n", argv[1]);
  printf("Fall  Cm/Cm0   Dev  Dist(m)\n");
  sprintf(path, "%s/ohne", argv[1]);
  getMaximum(path, &x, &y, &c0, &d0);
  printf("ohne   %5.2lf %5.2lf    %5.0lf\n", 
    1., d0, sqrt((x-Xq)*(x-Xq)+(y-Yq)*(y-Yq)));
  sprintf(path, "%s/lee", argv[1]);  
  getMaximum(path, &x, &y, &c, &d);
  printf("lee    %5.2lf %5.2lf    %5.0lf\n", 
    c/c0, d, sqrt((x-Xq)*(x-Xq)+(y-Yq)*(y-Yq)));
  sprintf(path, "%s/luv", argv[1]);  
  getMaximum(path, &x, &y, &c, &d);
  printf("luv    %5.2lf %5.2lf    %5.0lf\n", 
    c/c0, d, sqrt((x-Xq)*(x-Xq)+(y-Yq)*(y-Yq)));
  
  printf("\n");
  exit(0);

}

