//=================================================================== gencnv.c
//
// Conversion of *.def to *.arr
// ============================
//
// Copyright (C) Umweltbundesamt, 14191 Berlin, Germany, 2002-2005
// Copyright (C) Janicke Consulting, 26427 Dunum, Germany, 2002-2005
// email: info@austal2000.de
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of
// the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// last change 2003-04-02 lj
//
//==================================================================

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "IBJmsg.h"
#include "IBJary.h"
static char *eMODn = "GenCnv";

#include "genutl.h"
#include "genio.h"
#include "gencnv.h"

#define  BUFLEN  4000
#define  MAXFMT    20
#define  STRADD(a,b)  { int l; l=strlen(a); a[l]=b; a[l+1]='\0'; }

#define I Ii[0]
#define J Ii[1]
#define K Ii[2]
#define L Ii[3]

enum DATATYP {
  notanum, byteint, shrtint, dfltint, longint, shrtflt, longflt };

typedef struct {
  char name[20];
  char format[20];
  float faktor;
  enum DATATYP datyp;
  int elmsz;
  int offset; }  FMTREC;

/*============================ PROTOTYPES =============================*/

static long GetFormat(       /*  Format einlesen.                             */
void );

static void PrnFormat(/*  Format-Tabelle zum Testen ausdrucken.        */
void );

static int ElmPos(    /*  Position eines Elementes in der Struktur.    */
char *name );         /*  Name des Elementes.                          */
                      /*  RETURN: negativ, falls nicht gefunden.       */

static void Skalierung(/*  Float-Zahl durch den Faktor dividieren.      */
void *pv,             /*  Pointer auf die Variable.                    */
enum DATATYP typ,     /*  Daten-Typ der Variablen.                     */
float fak );          /*  Faktor.                                      */

static long Read1d(   /*  1-dimensionale Darstellung einlesen.         */
char *pc );           /*  Pointer auf erstes Token (Index-Festlegung). */

static long Read2d(   /*  2-dimensionale Darstellung einlesen.         */
char *pc );           /*  Pointer auf erstes Token (Index-Festlegung). */

static long PrnData(  /*  Eingelesene Daten zum Testen ausdrucken.     */
void );

/*=====================================================================*/

static FMTREC FmtTab[MAXFMT];
static int PosTab[MAXFMT];
static int Ii[4];

static char *Header, *Buf;
static int I1, I2, J1, J2, K1, K2, L1, L2, N;
static int NumItem, TtlSize, NumDim;
static float Faktor = 1;
static int FF[1000];
static ARYDSC *Pdsc;
static char Format[100];
static char CnvLabel = 'T';

/*================================================================== GetFormat
*/
static long GetFormat(              /*  Format einlesen.      */
void )
  {
  dP(GetFormat);
  int i, n, l;
  long rc;
  char *pc, *pp, *pf, *p, *ps, lc;
  FMTREC *prec;
  enum DATATYP tp;
  pc = Format;
  rc = -1;
  for ( n=0; n<MAXFMT; n++) {
    prec = FmtTab + n;
    p = strchr( pc, '%');
    if (p == NULL) { rc = 0;  break; }
    pp = p;
    l = pp - pc;
    strncpy( prec->name, pc, l );  prec->name[l] = 0;
    ps = prec->name;  while(*ps) *ps++ = toupper(*ps);
    prec->faktor = Faktor;                                //-00-03-10
    p++;
    if (*p == '(') {
      if (*++p != '*')  break;
      p = strchr( p, ')' );
      if (p == NULL) break;
      p++;
      if (pp[3] == ')') /* prec->faktor = Faktor */;      //-00-03-10
      else {
        i = sscanf( pp+3, "%f", &prec->faktor );
        if (i < 1)  break; }
      }
    pp = p;
    strcpy( prec->format, "%" );
    p = strpbrk( pp, "cefdx%" );
    if (p == NULL)  break;
    if (*p == '%')  break;
    pf = p;
    lc = 0;
    if (strchr( "dx", *pf ))
      switch (pf[-1]) {
        case 'b':  i = 1;              lc = '\0'; tp = byteint;  break;
        case 'h':  i = sizeof(short);  lc = 'h';  tp = shrtint;  break;
        case 'l':  i = sizeof(long);   lc = 'l';  tp = longint;  break;
        default:   i = sizeof(int);    tp = dfltint;
        }
    else
      if (*pf == 'c') {
        i = 1;
        tp = notanum;
        sscanf( pp, "%d", &i );
        if (i < 0)  i = -i;
        if (i < 1)  break; }
      else
        if (pf[-1] == 'l') {
          i = sizeof(double);  lc = 'l';  tp = longflt; }
        else { i = sizeof(float);  tp = shrtflt; }
    prec->elmsz = i;
    prec->datyp = tp;
    l = pf - pp + 1;
    i = (*pp == '-');
    /*                                                  -2003-04-02 lj
    for ( ; i<l; i++)
      if (isdigit(pp[i]))  STRADD( prec->format, pp[i] )
      else  break;
    */
    if (lc)  STRADD( prec->format, lc );
    STRADD( prec->format, *pf );
    pc = pf + 1;
    }
  if ((rc < 0) || (n == 0))                             eX(1);
  NumItem = n;
  TtlSize = 0;
  for (n=0; n<NumItem; n++) {
    FmtTab[n].offset = TtlSize;
    TtlSize += FmtTab[n].elmsz; }
  return NumItem;
eX_1:
  if (p)  *p = 0;
  eMSG("invalid structure of format %s!", pc);
}

/*================================================================== PrnFormat
*/
static void PrnFormat(/*  Format-Tabelle zum Testen ausdrucken.        */
void )
  {
  int n;
  Printf("\nformat (%s):\n", Format );
  for (n=0; n<NumItem; n++)
    Printf("%2d: %20s %8s / *%10.1e  +%d(%d)  ++%d\n",
           n, FmtTab[n].name, FmtTab[n].format, FmtTab[n].faktor,
           FmtTab[n].elmsz, FmtTab[n].datyp, FmtTab[n].offset );
  Printf("total size = %d\n", TtlSize );
  return;
  }

/*===================================================================== ElmPos
*/
static int ElmPos(    /*  Position eines Elementes in der Struktur.    */
char *name )          /*  Name des Elementes.                          */
  {                   /*  RETURN: negativ, falls nicht gefunden.       */
  int i;
  char *ps;
  ps = name;  while(*ps) *ps++ = toupper(*ps);
  for (i=0; i<NumItem; i++)
    if (0 == strcmp( name, FmtTab[i].name ))  return i;
  return -1;
  }

/*================================================================= Skalierung
*/
static void Skalierung(/*  Float-Zahl durch den Faktor dividieren.      */
void *pv,             /*  Pointer auf die Variable.                    */
enum DATATYP typ,     /*  Daten-Typ der Variablen.                     */
float fak )           /*  Faktor.                                      */
  {
  switch (typ) {
    case shrtflt: (*(float*)pv) /= fak;  break;
    case longflt: (*(double*)pv) /= fak;  break;
    default: ;
    }
  return;
  }

/*===================================================================== Read1d
*/
static long Read1d(   /*  1-dimensionale Darstellung einlesen.         */
char *pc )            /*  Pointer auf erstes Token (Index-Festlegung). */
  {
  dP(Read1d);
  int dv, iv, iv1, iv2, k, n, i, l, nn, pos, bval;
  long rc;
  void *pv;
  char *pse;
  dv = *pc - 'I';
  if (dv<0 || dv>3)                                     eX(1);
  iv1 = Pdsc->bound[dv].low;
  iv2 = Pdsc->bound[dv].hgh;
  if (NumItem == 1) {
    PosTab[0] = 0;
    n = 1; }
  else
    for (n=0; n<NumItem; n++) {
      pc = strtok(NULL, " |:");
      if (pc == NULL) break;
      pos = ElmPos(pc);  if (pos < 0)                   eX(2);
      PosTab[n] = pos;
      }
  nn = MIN( n, NumItem );
  for (iv=iv1; ; iv++) {
    N++;
    rc = GetLine(CnvLabel, Buf, BUFLEN);                eG(3);
    if (rc <= 0)  break;
    l = strlen(Buf) - 1;
    if (l>=0 && Buf[l]<' ' && Buf[l]>0)  Buf[l] = ' ';
    pc = strtok(Buf, " |:");  if (!pc)                  eX(4);
    k = sscanf(pc, "%d", &i);  if (k < 1)               eX(5);
    if (i<iv1 || i>iv2)                                 eX(6);
    Ii[dv] = i;
    pse = ChrPtr(Pdsc, I, J, K, L);  if (!pse)          eX(7);
    for (n=0; n<nn; n++) {
      pc = strtok(NULL, " |:");  if (!pc)               eX(8);
      pos = PosTab[n];
      pv = FmtTab[pos].offset + pse;
      if (FmtTab[pos].elmsz == 1) {
        k = sscanf(pc, FmtTab[pos].format, &bval);
        if (k == 1)  *(char*)pv = bval;
        }
      else  k = sscanf(pc, FmtTab[pos].format, pv);
      if (k < 1)                                        eX(9);
      Skalierung(pv, FmtTab[pos].datyp, FmtTab[pos].faktor);
      }
    }
  return 0;
eX_1:
  eMSG("invalid index identifier \"%c\"!", *pc);
eX_2:
  eMSG("missing item in tab header!");
eX_3:
  eMSG("error reading tab record %d!", N);
eX_4:
  eMSG("missing item in tab record %d!", N);
eX_5:
  eMSG("missing index value in tab record %d!", N);
eX_6:
  eMSG("index value %d not within range [%d,%d]!", i, iv1, iv2);
eX_7:
  eMSG("index (%d,%d,%d,%d) out of range!", I, J, K, L);
eX_8:
  eMSG("missing value in tab record %d!", N);
eX_9:
  eMSG("can't read value \"%s\" in tab record %d!", pc, N);
  }

/*===================================================================== Read2d
*/
static long Read2d(   /*  2-dimensionale Darstellung einlesen.         */
char *pc )            /*  Pointer auf erstes Token (Index-Festlegung). */
  {
  dP(Read2d);
  int dh, ih, ih1, ih2, dv, iv, iv1, iv2, n;
  int i, j, k, l, nsp, off, bval;
  long rc;
  char *ps, fmt[40];
  void *pv;
  enum DATATYP typ;
  float fak;
  dv = *pc - 'I';
  if ((dv < 0) || (dv >= NumDim))                       eX(1);
  iv1 = Pdsc->bound[dv].low;
  iv2 = Pdsc->bound[dv].hgh;
  dh = pc[2] - 'I';
  if ((dh < 0) || (dh >= NumDim))                       eX(2);
  ih1 = Pdsc->bound[dh].low;
  ih2 = Pdsc->bound[dh].hgh;
  for ( nsp=0; ; nsp++) {
    pc = strtok( NULL, " |:" );
    if (pc == NULL) break;
    j = strtol( pc, &ps, 0 );
    if (j<ih1 || j>ih2)                                 eX(3);
    FF[nsp] = j; }
  for (iv=iv1; ; iv++) {
    N++;
    rc = GetLine(CnvLabel, Buf, BUFLEN);                        eG(4);
    if (rc <= 0)  break;
    l = strlen(Buf) - 1;
    if (l>=0 && Buf[l]<' ' && Buf[l]>0)  Buf[l] = ' ';
    pc = strtok(Buf, " |:\t\r");  if (!pc)              eX(5);
    k = sscanf(pc, "%d", &i);
    if (k < 1)                                          eX(6);
    if (i<iv1 || i>iv2)                                 eX(7);
    Ii[dv] = i;
    if (NumItem == 1)
      for (j=0; j<nsp; j++) {
        pc = strtok(NULL, " |:\t\r");  if (!pc)         eX(8);
        ih = FF[j];
        Ii[dh] = ih;
        pv = AryPtrX(Pdsc, I, J, K, L);
        if (FmtTab[0].elmsz == 1) {
          k = sscanf(pc, FmtTab[0].format, &bval);
          if (k == 1)  *(char*)pv = bval;
          }
        else  k = sscanf(pc, FmtTab[0].format, pv);
        if (k < 1)                                      eX(9);
        Skalierung(pv, FmtTab[0].datyp, FmtTab[0].faktor);
        }
    else
      for (n=0; n<NumItem; n++) {
        N++;
        rc = GetLine('+', Buf, BUFLEN);  if (rc < 0)    eX(10);
        if (rc == 0)  break;
        pc = strtok(Buf, " |:");  if (!pc)              eX(11);
        i = ElmPos(pc);  if (i < 0)                     eX(12);
        strcpy(fmt, FmtTab[i].format);
        off = FmtTab[i].offset;
        typ = FmtTab[i].datyp;
        fak = FmtTab[i].faktor;
        for (j=0; j<nsp; j++) {
          pc = strtok(NULL, " |:");  if (!pc)           eX(13);
          ih = FF[j];
          Ii[dh] = ih;
          pv = off + ChrPtr(Pdsc, I, J, K, L);
          if (FmtTab[i].elmsz == 1) {
            k = sscanf(pc, fmt, &bval);
            if (k == 1)  *(char*)pv = bval;
            }
          else  k = sscanf(pc, fmt, pv);
          if (k < 1)                                    eX(14);
          Skalierung(pv, typ, fak);
          }
        }
    }
  return 0;
eX_1:
  eMSG("invalid identifier \"%c\" for first index!", pc[0]);
eX_2:
  eMSG("invalid identifier \"%c\" for second index!", pc[2]);
eX_3:
  eMSG("column index %d not within range [%d,%d]!", j, ih1, ih2);
eX_4:
  eMSG("error reading record %d!", N);
eX_5:
  eMSG("index not found in record %d!", N);
eX_6:
  eMSG("can't read index in record %d!", N);
eX_7:
  eMSG("row index %d not within range [%d,%d]!", i, iv1, iv2);
eX_8:
  eMSG("no value in column %d record %d!", j+1, N);
eX_9:
  eMSG("can't read value in column %d record %d!", j+1, N);
eX_10:
  eMSG("no definition of component %d in record %d gefunden!", n+1, N);
eX_11:
  eMSG("index not found in record %d!", N);
eX_12:
  eMSG("can't read index in record %d!", N);
eX_13:
  eMSG("no value in column %d record %d!", j+1, N);
eX_14:
  eMSG("can't read value \"%s\" in column %d record %d!", pc, j+1, N);
  }

/*==================================================================== PrnData
*/
static long PrnData(/*  Eingelesene Daten zum Testen ausdrucken.     */
void )
  {
  dP(PrnData);
  int i, j, k, l, n;
  void *pv;
  for (i=I1; i<=I2; i++)
    for (j=J1; j<=J2; j++)
      for (k=K1; k<=K2; k++)
        for (l=L1; l<=L2; l++) {
          Printf("(%d,%d,%d,%d)", i, j, k, l );
          for (n=0; n<NumItem; n++) {
            Printf( " %s", FmtTab[n].name );
            pv = AryPtr(Pdsc,i,j,k,l);  if (!pv)                        eX(1);
            pv = FmtTab[n].offset + (char*)pv;
            Printf("(%lx):", pv);
            if (FmtTab[n].elmsz == 2)
              Printf( FmtTab[n].format, *(int*)pv );
            if (FmtTab[n].elmsz == 4) {
              if (strpbrk( FmtTab[n].format, "ef" ))
                Printf( FmtTab[n].format, *(float*)pv );
              if (strpbrk( FmtTab[n].format, "dx" ))
                Printf( FmtTab[n].format, *(long*)pv ); }
            }
          Printf( "\n" );
          }
  return 0;
eX_1:
  eMSG("range error (%d,%d,%d,%d)!", i, j, k, l);
  }

/*===================================================================== GenCnv
*/
#define  GET(a,b,c,d)  GetData((a),(b),(c),(d)); strcpy(nn,(a)); eG(99)

long GenCnv( char *name, ARYDSC *pdsc, char *hdr, int prnflag )
  {
  dP(GenCnv);
  char *pc, *ps, nn[40];
  long rc;
  int l;
  rc = OpenInput(name, NULL);                           eG(1);
  if (rc < 0)  return -1;
  Header = hdr;
  Pdsc = pdsc;
  Buf = ALLOC(BUFLEN);  if (!Buf)                       eX(2);
  N++;
  rc = GetLine('.', Header, BUFLEN);  if (rc < 0)       eX(3);
  rc = GET("NUMDIM", Header, "%d", &NumDim);
  if (rc == 0)  NumDim = 2;
  if ((NumDim < 1) || (NumDim > 4))                     eX(4);
  GET("I",  Header, "%d", &I);
  GET("I1", Header, "%d", &I1);
  GET("I2", Header, "%d", &I2);
  GET("J",  Header, "%d", &J);
  GET("J1", Header, "%d", &J1);
  GET("J2", Header, "%d", &J2);
  GET("K",  Header, "%d", &K);
  GET("K1", Header, "%d", &K1);
  GET("K2", Header, "%d", &K2);
  GET("L",  Header, "%d", &L);
  GET("L1", Header, "%d", &L1);
  GET("L2", Header, "%d", &L2);
  Faktor = 1;
  GET("FAKTOR|fact|factor", Header, "%f", &Faktor);
  rc = GET("FORMAT|form", Header, "%s", Format);
  if (rc == 0) {
    NumItem = 1;
    TtlSize = 4;
    *FmtTab[0].name = 0;
    strcpy( FmtTab[0].format, "%e" );
    FmtTab[0].faktor = Faktor;
    FmtTab[0].elmsz = 4;
    FmtTab[0].offset = 0;
    FmtTab[0].datyp = shrtflt;
    }
  else {
    GetFormat();                                        eG(5);
    }
  AryCreate(Pdsc, TtlSize, NumDim, I1, I2, J1, J2, K1, K2, L1, L2);     eG(12);
  pc = (char *) Pdsc->start;
  memset(pc, 0, Pdsc->ttlsz);
  while (1) {
    N++;
    rc = GetLine('!', Buf, BUFLEN);  /* if (rc < 0)     eX(6); -25aug95-*/
    if (rc > 0) {
      if (prnflag)  Printf("!%s\n", Buf);
      l = strlen(Buf) - 1;
      if (l>=0 && Buf[l]<' ' && Buf[l]>0)  Buf[l] = ' ';
      pc = strtok(Buf, " :|");  if (!pc)                eX(7);
      ps = pc;  while(*ps) *ps++ = toupper(*ps);
      ps = strchr(pc, '\\');
      if (!ps) {
        Read1d(pc);                                     eG(8);
        }
      else {
        if (NumDim < 2)                                 eX(17);
        Read2d(pc);                                     eG(9);
        }
      }
    if (*Buf != '.') break;
    N++;
    rc = GetLine('.', Buf, BUFLEN);  if (rc < 0)        eX(10);
    if (prnflag) Printf(".%s\n", Buf);
    GET("I", Buf, "%d", &I);
    GET("J", Buf, "%d", &J);
    GET("K", Buf, "%d", &K);
    GET("L", Buf, "%d", &L);
    }
  if (*Buf > ' ') {
    Printf("\nGENCNV:unexpected input:\n>>>%s<<<\n", Buf);
    }
  CloseInput();                                         eG(11);
  FREE(Buf);
  return 0;
eX_99:
  eMSG("error reading %s!", nn);
eX_1:
  eMSG("can't read file %s!", name);
eX_2:
  eMSG("no memory for buffer allocation!");
eX_3:
  eMSG("no header found in file %s!", name);
eX_4:
  eMSG("invalid number of dimensions (=%d) in file %s!", NumDim, name);
eX_5:
  eMSG("can't analyse format specifier in file %s!", name);
eX_7: eX_17:
  eMSG("error parsing tab-header in file %s!", name);
eX_8:
  eMSG("error reading 1d tab in file %s!", name);
eX_9:
  eMSG("error reading 2d tab in file %s!", name);
eX_10:
  eMSG("error searching for new header section in file %s!", name);
eX_11:
  eMSG("error closing input file %s!", name);
eX_12:
  eMSG("can't create array of file %s!", name);
  }

#ifdef MAIN /*##############################################################*/

ARYDSC Dsc;
char Hdr[GENHDRLEN];

void MyExit(void)
  {
  if (MsgCode < 0)  Printf("ERROR!\n");
  }

/*======================================================================= main
*/
int main( int argc, char *argv[] )
  {
  dP(GenCnvMain);
  char name[256], s[120], *pc, newname[256];
  long rc;
  int i, n;
  atexit(MyExit);
  Printf("GENCNV: convert text file to binary file (.arr)\n");
  n = 0;
  *newname = 0;
  for (i=1; i<argc; i++) {
    strcpy(s, argv[i]);
    if (s[0] == '-') {
      switch (s[1]) {
        case 'l': CnvLabel = s[2];
                  break;
        case 'o': strcpy(newname, s+2);
                  break;
        default:  ;
      }
      continue;
    }
    strcpy(name, s);
    rc = GenCnv(name, &Dsc, Hdr, 1);                    eG(1);
    if (rc < 0)                                         eX(10);
    Printf("input file %s closed.\n", name);
    if (*newname)  strcpy(name, newname);
    pc = strrchr(name, '.');
    if (!pc)  strcat(name, ".arr");
    else  strcpy(pc, ".arr");
    ArrWrite(name, Hdr, &Dsc);                          eG(2);
    Printf("file %s written.\n", name);
    n++;
  }
  Printf("program GENCNV finished.\n");
  return 0;
eX_10:
  nMSG("can't open file %s!", name);
  return 1;
eX_1:
  nMSG("file %s not converted!", name);
  return 2;
eX_2:
  nMSG("can't write file %s!", name);
  return 3;
  }
#endif
/*==========================================================================*/
