// ======================================================== TALprairie.c
//
// Read prairie grass data, write input for austal2000
// ===================================================
//
// Copyright (C) Janicke Consulting, Dunum, Germany, 2002
// email: info@janicke.de
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of
// the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// last change:  2002-09-27  lj
//
//========================================================================

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

static int Nv, Nn, Ic[101];
static int N1=7, N2=10, Ll;
static char Line[4000], *Names[100], *Header, *Data, Stoff[40]="so2";
static char Path[256];
static float Z0=0.008, Hq=0.46, Ha=2;

static float X0=-32.5, Y0=-502.5, Dd=5;
static int Nx=185, Ny=201, Kref=7;
static int set_hm = 0;

static void Trim(        // Blanks am Anfang und Ende beseitigen
unsigned char *t )       // zu trimmende Zeichenkette
  {
  int l, ll, i;
  ll = strlen( (char*)t );
  for (l=ll-1; l>=0; l--)  if (t[l] != ' ')  break;
  t[l+1] = 0;
  ll = l+1;
  for (l=0; l<ll; l++)  if (t[l] > ' ') break;
  if (l > 0) {
    for (i=0; i<ll-l; i++)  t[i] = t[i+l];
    t[i] = 0; }
  return; }

//============================================================== ReadPrairie
int ReadPrairie( char *name )
  {
  FILE *f;
  char fn[256];
  int l, i, n;
  sprintf(fn, "%s/%s", Path, name);
  f = fopen(fn, "rb");
  if (!f) {
    printf("Datei %s nicht lesbar!\n", fn);
    exit(1);
    }
  if (!fgets(Line, 4000, f))  return -1;
  Nn = 0;
  l = strlen(Line);
  Ll = l;
  for (i=0; i<l; i++)
    if (Line[i] == '.') {
      Nn++;
      if (Nn > 100)  return -2;
      Ic[Nn] = i;
      }
  for (i=1; i<N1; i++) {
    if (!fgets(Line, 4000, f))  return -3;
    l = strlen(Line);
    if (l > Ll)  Ll = l;
    }
  Header = malloc(l+1);  if (!Header)  return -4;
  strcpy(Header, Line);
  Ic[0] = -1;
  for (n=0; n<Nn; n++) {
    *Line = 0;
    Names[n] = Header+Ic[n]+1;
    strncat(Line, Names[n], Ic[n+1]-Ic[n]);
    Trim(Line);
    strcpy(Names[n], Line);
    }
  for (i=N1; i<N2-1; i++) {
    if (!fgets(Line, 4000, f))  return -5;
    l = strlen(Line);
    if (l > Ll)  Ll = l;
    }
  Nv = 0;
  while (fgets(Line, 4000, f)) {
    l = strlen(Line);
    if (l > Ll)  Ll = l;
    Trim(Line);
    if (*Line)  Nv++;
    }
  fseek(f, 0, SEEK_SET);
  Ll++;
  Data = malloc(Ll*Nv);  if (!Data)  return -6;
  for (i=0; i<N2-1; i++)
    if (!fgets(Line, 4000, f))  return -7;
  for (i=0; i<Nv; ) {
    if (!fgets(Data+i*Ll, Ll, f))  return -7;
    strcpy(Line, Data+i*Ll);
    Trim(Line);
    if (*Line)  i++;
    }
  fclose(f);
  return Nv;
  }

static int Value( int row, char *name, int seq, float *pv )
  {
  int i, n;
  char s[256];
  i = 0;
  *pv = 0;
  for (n=0; n<Nn; n++)
    if (!strcmp(Names[n], name)) {
      i++;
      if (i >= seq)  break;
      }
  if (n >= Nn)  return -1;
  *s = 0;
  strncat(s, Data+row*Ll+Ic[n]+1, Ic[n+1]-Ic[n]);
  Trim(s);
  if (!*s)  return 0;
  *pv = atof(s);
  return 1;
  }

//=============================================================== WriteAustal
int WriteAustal( void )
  {
  int i, iv, ii, nd, j;
  float xp, ua, lm, hm;
  FILE *f;
  char fn[256];
  sprintf(fn, "%s/%s", Path, "austal2000.txt");
  f = fopen(fn, "wb");
  if (!f) {
    printf("Datei %s nicht beschreibbar!\n", fn);
    return -1;
    }
  fprintf(f, "-- Eingabe-Datei fr Austal2000: Prairie Grass\n");
  fprintf(f, "-- von TALprairie erstellt.\n");
  fprintf(f, "-- \n");
  fprintf(f, "ti  \"Nachrechnung der Prairie Grass Experimente\"\n");
  fprintf(f, "os  \"NOSTANDARD;Kref=%d;\"\n", Kref);
  fprintf(f, "qs  %d\n", 6);
  fprintf(f, "dd  %1.1f\n", Dd);
  fprintf(f, "x0  %1.1f\n", X0);
  fprintf(f, "y0  %1.1f\n", Y0);
  fprintf(f, "nx  %d\n", Nx);
  fprintf(f, "ny  %d\n", Ny);
  fprintf(f, "hh  0 0.2 0.4 0.6 0.8 1 1.3 1.7 2 2.5 3 4 5 6 7 8 9 10"
             " 12 15 20 30 40 50 70 100 200 300 400 500 600 700 800"
             " 900 1000 1200 1400 1800 2000\n");
  fprintf(f, "xq  0\n");
  fprintf(f, "yq  0\n");
  fprintf(f, "hq  %2.2f\n", Hq);
  if (set_hm)  fprintf(f, "hm  ?\n");                           //-2002-08-01
  fprintf(f, "nh3 ?\n");
  fprintf(f, "so2 ?\n");
  fprintf(f, "z0  %1.3f\n", Z0);
  fprintf(f, "ha  %1.1f\n", Ha);
  fprintf(f, "-------------------------------------------\n");
  fclose(f);
  //
  iv = Nv-1;
  i = Value(iv, "EXP", 0, &xp);
  if (i < 1) {
    printf("Zeile %d: EXP nicht lesbar!\n", iv);
    return -2;
  }
  nd = 24*(int)xp;
  sprintf(fn, "%s/%s", Path, "zeitreihe.dmna");
  f = fopen(fn, "wb");
  if (!f) {
    printf("Datei %s nicht beschreibbar!\n", fn);
    return -1;
    }
  fprintf(f, "-- Zeitreihe fr Austal2000: Prairie Grass\n");
  fprintf(f, "-- von TALprairie erstellt.\n");
  fprintf(f, "-- \n");
  if (set_hm) 
    fprintf(f, "form \"%s\" \"%s\" \"%s\" \"%s\" \"%s\" \"%s\" \"%s\"\n",
    "te%20lt", "ra%5.0f", "ua%5.1f", "lm%7.1f", "hm%7.1f", "01.nh3%10.3e", "01.so2%10.3e");
  else
    fprintf(f, "form \"%s\" \"%s\" \"%s\" \"%s\" \"%s\" \"%s\"\n",
    "te%20lt", "ra%5.0f", "ua%5.1f", "lm%7.1f", "01.nh3%10.3e", "01.so2%10.3e");
  fprintf(f, "mode \"%s\"\n", "text");
  fprintf(f, "dims %d\n", 1);
  fprintf(f, "lowb %d\n", 1);
  fprintf(f, "hghb %d\n", nd);
  fprintf(f, "sequ \"%s\"\n", "i");
  fprintf(f, "*\n");
  xp = -1;
  for (iv=0, ii=1; iv<Nv; ii++) {
    if (xp < ii) {
      i = Value(iv, "EXP", 0, &xp);
      if (i < 1) {
        printf("Zeile %d: EXP nicht lesbar!\n", iv);
        return -2;
      }
      if (xp < ii)  return -3;
      i = Value(iv, "U", 0, &ua);
      if (i < 1) printf("Zeile %d: U nicht lesbar!\n", iv);
      i = Value(iv, "L", 0, &lm);
      if (i < 1) printf("Zeile %d: L nicht lesbar!\n", iv);
      i = Value(iv, "H", 0, &hm);
      if (i < 1)  hm = 0;
      iv++;
    }
    for (j=0; j<24; j++) {
      fprintf(f, "  %3d.%02d:00:00  270  %5.1f", ii-1, j+1, ua);
      fprintf(f, "  %7.1f", (xp==ii) ? lm : 0.0);
      if (set_hm)  fprintf(f, "  %7.1f", (xp==ii) ? hm : 0.0);      //-2002-09-27
      fprintf(f, "  %5.1f",   (j==0) ? 24.0 : 0.0);
      fprintf(f, "  %5.1f\n", (j==0) ? 24.0 : 0.0);
    }
  }
  fprintf(f, "***\n");
  fclose(f);
  return 0;
}

//============================================================== EvaluateAustal
int EvaluateAustal( int day, float xx[], int nx, float result[][3] ) {
  int i, j, n;
  int buflen=20*Nx;
  char fn[256], *buf, *pc;
  float *cc, cm, cy, sy, c, x, y, ym, yq;
#define CC(i,j)  cc[Ny*(i-1)+j-1]
  FILE *f;
  sprintf(fn, "%s/%s-%03dz.dmna", Path, Stoff, day);
  f = fopen(fn, "rb");
  if (!f) {
    printf("Datei %s nicht gefunden!\n", fn);
    return -2;
  }
  buf = malloc(buflen); if (!buf)  return -3;
  while (fgets(buf, buflen, f)) {
    if (*buf == '*')  break;
  }
  if (*buf != '*')  return -4;
  cc = malloc(Nx*Ny*sizeof(float));  if (!cc)  return -5;
  for (j=Ny; j>=1; j--) {
    if (!fgets(buf, buflen, f))  return -6;
    pc = strtok(buf, " \t\r\n");
    for (i=1; i<=Nx; i++) {
      if (!pc)  return -7;
      if (1 > sscanf(pc, "%f", &c))  return -8;
      CC(i,j) = c;
      pc = strtok(NULL, "\t\r\n");
    }
  }
  fclose(f);
  free(buf);
  for (n=1; n<=nx; n++) {
    x = xx[n];
    i = 1 + (x-X0)/Dd;
    cm = 0;
    cy = 0;
    sy = 0;
    ym = 0;
    yq = 0;
    for (j=1; j<=Ny; j++) {
      c = CC(i,j);
      cy += c;
      if (c > cm)  cm = c;
      y = Y0 + Dd*(j-0.5);
      ym += c*y;
    }
    if (cy == 0)  sy = 0;
    else {
      ym /= cy;
      for (j=1; j<=Ny; j++) {
        c = CC(i,j);
        y = Y0 + Dd*(j-0.5);
        yq += c*(y-ym)*(y-ym);
      }
      sy = sqrt(yq/cy);
      cy *= Dd;
    }
    result[n][0] = sy;
    result[n][1] = cy;
    result[n][2] = cm;
  }
  free(cc);
#undef CC
  return 0;
}

//============================================================= CompareAustal
int CompareAustal( void )
  {
  int i, k, iv, m, day;
  float xp, us, lmo, cy, sym, cm, sya, cyl, syl, cml;
  float u, x, w2p, a, result[6][3];
  char ffn[3][256];
  static char *nn[3] = { "Fahnenbreite Sigma-y (m)", "Konzentrationsintegral 1.e3*u*Cy/Q (1/m)",
    "Maximale Konzentration 1.e6*u*Cmax/Q (1/m^2)" };
  FILE *ff[3];
  static float xx[6] = { 0, 50, 100, 200, 400, 800 };
  sprintf(ffn[0], "%s/vergleich-%s-%s.txt", Path, Stoff, "sy");
  sprintf(ffn[1], "%s/vergleich-%s-%s.txt", Path, Stoff, "cy");
  sprintf(ffn[2], "%s/vergleich-%s-%s.txt", Path, Stoff, "cm");
  for (m=0; m<3; m++) {
    ff[m] = fopen(ffn[m], "w");
    if (!ff[m]) {
      printf("Datei %s kann nicht angelegt werden!\n", ffn[m]);
      return -4;
      }
    fprintf(ff[m], "%s: %s\n\n N XP 1000/L      L   U*   U2", Stoff, nn[m]);
    }
  fprintf(ff[0], " SYM050 SYA050 SYR050 SYM100 SYA100 SYR100 SYM200 SYA200 SYR200"
                 " SYM400 SYA400 SYR400 SYM800 SYA800 SYR800\n");
  fprintf(ff[1], " CYM050 CYR050 CYM100 CYR100 CYM200 CYR200"
                 " CYM400 CYR400 CYM800 CYR800\n");
  fprintf(ff[2], "   CMM050   CMR050   CMM100   CMR100   CMM200   CMR200"
                 "   CMM400   CMR400   CMM800   CMR800\n");
  w2p = sqrt(2*3.1415926);
  for (iv=0; iv<Nv; iv++) {
    i = Value(iv, "EXP", 0, &xp);
    if (i < 1) { printf("Zeile %d: EXP nicht lesbar! (%d)\n", iv+1, i);  return -1; }
    i = Value(iv, "L", 0, &lmo);
    if (i < 1) { printf("Zeile %d: L nicht lesbar! (%d)\n", iv+1, i);  return -1; }
    i = Value(iv, "U", 0, &u);
    if (i < 1) { printf("Zeile %d: U nicht lesbar! (%d)\n", iv+1, i);  return -1; }
    i = Value(iv, "U*", 0, &us);
    if (i < 1) { printf("Zeile %d: U* nicht lesbar! (%d)\n", iv+1, i);  return -1; }
    day = (int)xp;
    i = EvaluateAustal(day, xx, 5, result);
    if (i < 0) {
      printf("Fehler %d bei der Auswertung von Tag %d\n", -i, day);
      return -2;
      }
    for (m=0; m<3; m++)
      fprintf(ff[m], "%2d %2.0f %6.1f %6.1f %4.2f %4.1f", iv+1, xp, 1000/lmo, lmo, us, u);
    a = u*1.e-3;
    for (k=1; k<=5; k++) {
      x = xx[k];
      i = Value(iv, "C/Q",  k, &cm);
      cm = (i == 1) ? cm*1.e9 : 0;
      i = Value(iv, "Cy/Q", k, &cy);
      cy = (i == 1) ? cy*1.e6 : 0;
      i = Value(iv, "SIGY", k, &sym);
      if (i < 1)  sym = 0;
      sya = (cm > 0) ? 1.e3*cy/(w2p*cm) : 0;
      syl = result[k][0];
      cyl = result[k][1];
      cml = result[k][2]*1.e3;
      fprintf(ff[0], " %6.1f %6.1f %6.1f", sym, sya, syl);
      fprintf(ff[1], " %6.2f %6.2f", a*cy, a*cyl);
      fprintf(ff[2], " %8.2f %8.2f", a*cm, a*cml);
      }
    for (m=0; m<3; m++)  fprintf(ff[m], "\n");
    }
  for (m=0; m<3; m++)  fclose(ff[m]);
  return 0;
  }

//=============================================================== main
int main( int argc, char *argv[] )
  {
  int i, n, write_austal=0, compare_austal=0;
  char s[256], *pc;
  if (argc < 2) {
    printf("Aufruf: TALprairie <Pfad> <Option> ... \n");
    printf("Option: -a<Stoff>        Ergebnisse von AUSTAL2000 analysieren\n");
    printf("        -e               Dateien fr AUSTAL2000 erstellen\n");
    exit(0);
  }
  for (i=1; i<argc; i++) {
    strcpy(s, argv[i]);
    if (*s == '-')
      switch (s[1]) {
        case 'a': compare_austal = 1;
                  strcpy(Stoff, s+2);
                  break;
        case 'e': write_austal = 1;
                  break;
        default:  ;
      }
    else  strcpy(Path, s);
  }
  for (pc=Path; (*pc); pc++) if (*pc == '\\')  *pc = '/';
  n = ReadPrairie("prairie.txt");
  if (n < 0) {
    printf("Fehler %d beim Einlesen der Prairie Grass Daten!\n", -n);
    exit(2);
  }
  if      (write_austal)   n = WriteAustal();
  else if (compare_austal) n = CompareAustal();
  else {
    printf("Keine Aufgabe fr TALprairie angegeben!\n");
    n = 0;
  }
  if (n < 0)  printf("Fehler %d aufgetreten!\n", -n);
  printf("Programm TALprairie beendet.\n");
  return -n;
}

